<?php
use OrderDetect\Helper;

if ( ! function_exists( 'get_orders_by_phone_number' ) ) {
    function get_orders_by_phone_number( $phone_number ) {
        
        $orders_data = [];
        $cached_order_ids = get_post_meta($phone_number, $phone_number . '_multi_order', true);
        if (!empty($cached_order_ids) && is_array($cached_order_ids)) {
            foreach ($cached_order_ids as $order_id) {
                $order = wc_get_order($order_id);
                if ($order) {
                    $orders[] = $order;
                    $orders_data[] = $order_id;
                }
            }
        }

        // If no cached data, fall back to database queries
        global $wpdb;

        $query_one = "
            SELECT posts.ID
            FROM {$wpdb->prefix}posts AS posts
            INNER JOIN {$wpdb->prefix}postmeta AS meta ON posts.ID = meta.post_id
            WHERE posts.post_type = 'shop_order'
            AND posts.post_status IN ('wc-pending', 'wc-processing', 'wc-on-hold', 'wc-completed', 'wc-cancelled', 'wc-refunded', 'wc-failed')
            AND meta.meta_key = '_billing_phone'
            AND meta.meta_value LIKE %s
            ORDER BY posts.ID DESC
        ";

        $query_two = "
            SELECT posts.ID
            FROM {$wpdb->prefix}posts AS posts
            INNER JOIN {$wpdb->prefix}wc_order_addresses AS meta ON posts.ID = meta.order_id
            WHERE posts.post_type = 'shop_order_placehold'
            AND meta.address_type = 'billing'
            AND meta.phone LIKE %s
            ORDER BY posts.ID DESC
        ";

        if( empty( $orders_data ) ){
            $orders_data = search_phone_number_multiple_order( $query_one, $phone_number );
        }
        if( empty( $orders_data ) ) {
            $orders_data = search_phone_number_multiple_order( $query_two, $phone_number );
        } 

        $orders = [];
        $order_ids_to_cache = [];
        foreach ($orders_data as $order_data) {
            if (isset($order_data)) {
                $order = wc_get_order($order_data);
                if ($order) {
                    $orders[] = $order;
                    $order_ids_to_cache[] = $order->get_id();
                }
            }
        }

        // Cache the order IDs for future use
        if (!empty($order_ids_to_cache)) {
            update_post_meta($phone_number, $phone_number . '_multi_order', $order_ids_to_cache);
        }

        return $orders;
    }
}

if ( ! function_exists( 'search_phone_number_multiple_order' ) ) {
    function search_phone_number_multiple_order( $query, $phone_number ) {
        global $wpdb;

        $prepared_query = $wpdb->prepare($query, '%' . $wpdb->esc_like($phone_number) . '%');
        $order_ids = $wpdb->get_col($prepared_query);

        if ( empty( $order_ids ) ) {
            return [];
        }

        $placeholders = implode(', ', array_fill(0, count($order_ids), '%d'));
        $orders_query = "
            SELECT ID, post_author, post_date, post_date_gmt, post_content, post_title, post_excerpt,
                post_status, comment_status, ping_status, post_password, post_name, to_ping, pinged,
                post_modified, post_modified_gmt, post_content_filtered, post_parent, guid, menu_order,
                post_type, post_mime_type, comment_count
            FROM {$wpdb->prefix}posts
            WHERE ID IN ($placeholders)
            ORDER BY ID DESC
        ";
        $prepared_orders_query = $wpdb->prepare($orders_query, ...$order_ids);
        $orders_data = $wpdb->get_results($prepared_orders_query);

        return $orders_data;

    }
}

function normalize_customer_phone_number( $phone_number ) {
    $normalized = preg_replace('/[^\d+]/', '', $phone_number);
    if (substr($normalized, 0, 3) === '+88') {
        $normalized = substr($normalized, 3);
    } elseif (substr($normalized, 0, 2) === '88') {
        $normalized = substr($normalized, 2);
    }
    return $normalized;
}

if ( ! function_exists( 'remove_admin_notices_on_plugin_pages' ) ) {
    function remove_admin_notices_on_plugin_pages() {
        $current_screen = get_current_screen();
        
        if ($current_screen && in_array($current_screen->id, [
            'order-detect_page_sms-log',
            'order-detect_page_block-list',
            'toplevel_page_order-detect'
        ])) {
            remove_all_actions('admin_notices');
            remove_all_actions('all_admin_notices');
            remove_all_actions('user_admin_notices');
            remove_all_actions('network_admin_notices');
        }
    }
    add_action( 'in_admin_header', 'remove_admin_notices_on_plugin_pages', PHP_INT_MAX );
}


if ( ! function_exists( 'od_send_alerts_based_on_order_status' ) ) {
    function od_send_alerts_based_on_order_status($phone_number, $message) {

        global $odSmsProvider;
        $settings = wp_parse_args(get_option('orderdetect_settings', array(
            'sms_provider' => '',
            'sms_api_key' => array(
                'greenweb' => '',
                'alpha' => '',
                'dianahost' => '',
                'mimsms' => '',
                'bulksmsbd' => ''
            ),
            'mimsms_sender_id' => '',
            'mimsms_username' => '',
            'bulksmsbd_sender_id' => '',
            'enable_otp' => 0,
            'checkout_otp_message' => '',
            'dianahost_sender_id' => '',
            'enable_invoice' => 0,
            'enable_packing_slip' => 0,
            'invoice_packing_slip_logo' => '',
            'business_name' => '',
            'enable_footer_text' => 0,
            'footer_text_heading' => '',
            'footer_text_details' => '',
            'delivery_partner' => '',
            'primary_color' => '#000'
        )));
        $endpoint = $odSmsProvider[$settings['sms_provider']]; 
        $api_key = $settings["sms_api_key"][$settings['sms_provider']];
        $to = Helper::validate_and_format_phone_number( $phone_number );

        if( "greenweb" === $settings['sms_provider'] ) {
            $endpoint .= '/api.php';
            $data = [
                'to'=>"$to",
                'message'=>"$message",
                'token'=>"$api_key"
            ];
        } else if( "alpha" === $settings['sms_provider'] ) {
            $endpoint .= '/sendsms';
            $data = [
                'to'=>"$to",
                'msg'=>"$message",
                'api_key'=>"$api_key"
            ];
        } else if( "dianahost" === $settings['sms_provider'] ) {
            $endpoint .= '/sms/send';
            $data = [
                'recipient'=>"$to",
                'type' => 'plain',
                'sender_id' => $settings['dianahost_sender_id'],
                'message'=>"$message",
            ];
        } else if( "mimsms" === $settings['sms_provider'] ) {
            $endpoint .= '/SmsSending/Send';
            $data = [
                'UserName' => $settings['mimsms_username'],
                'Apikey' => "$api_key",
                'MobileNumber' => "$to",
                'SenderName' => $settings['mimsms_sender_id'],
                'TransactionType' => 'T',
                'Message' => "$message",
            ];
            
        } else if( "bulksmsbd" === $settings['sms_provider'] ) {
            $endpoint .= '/smsapi';
            $data = [
                'number'=>"$to",
                'type' => 'text',
                'api_key'=>"$api_key",
                'senderid' => $settings['bulksmsbd_sender_id'],
                'message'=>"$message",
            ];
        }

        if( "mimsms" === $settings['sms_provider'] ){
            file_get_contents($endpoint.'?'.http_build_query($data));
        } else {
            $ch = curl_init(); 
            curl_setopt($ch, CURLOPT_URL, $endpoint);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
            curl_setopt($ch, CURLOPT_ENCODING, '');
    
            if( "dianahost" === $settings['sms_provider'] ) {
                curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                    'Accept: application/json',
                    "Authorization: Bearer $api_key",
                    "Content-Type: application/json"
                ));
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            } else {
                curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
            }
    
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            $smsresult = curl_exec($ch);
    
            if ($smsresult === false) {
                error_log('cURL Error: ' . curl_error($ch));
            } else {
                error_log('cURL Response: ' . $smsresult);
            }
            curl_close($ch);
        }

        Helper::update_balance();
        Helper::send_sms_balance_notification();

    }
}

if ( ! function_exists( 'get_clean_price_with_currency' ) ) {
    function get_clean_price_with_currency($price) {
        return html_entity_decode(strip_tags(wc_price($price)));
    }
}

if ( ! function_exists( 'add_otp_enable_forcefully' ) ) {
    function add_otp_enable_forcefully(){
        $settings = wp_parse_args(get_option('orderdetect_settings', array(
            'sms_provider' => '',
            'sms_api_key' => array(
                'greenweb' => '',
                'alpha' => '',
                'dianahost' => '',
                'mimsms' => '',
                'bulksmsbd' => ''
            ),
            'mimsms_sender_id' => '',
            'mimsms_username' => '',
            'bulksmsbd_sender_id' => '',
            'enable_otp' => 0,
            'checkout_otp_message' => '',
            'dianahost_sender_id' => '',
            'enable_invoice' => 0,
            'enable_packing_slip' => 0,
            'invoice_packing_slip_logo' => '',
            'business_name' => '',
            'enable_footer_text' => 0,
            'footer_text_heading' => '',
            'footer_text_details' => '',
            'delivery_partner' => '',
            'primary_color' => '#000'
        )));
        if( 
            ( isset($settings['enable_otp']) && 1 == $settings['enable_otp'] ) ||
            ( get_option('od_phone_block_enabled') || get_option('od_device_block_enabled') || get_option('od_ip_block_enabled') ) ||
            ( ! empty(get_option('blocked_ips')) || ! empty(get_option('blocked_phones')) )
        ) {
        ?>
        <script>
            jQuery(document).ready(function($) {
                function updateButtonClass() {
                    var $button = $('button[name="woocommerce_checkout_place_order"]');
                    if ($button.length) {
                        if (!$button.hasClass('button') || !$button.hasClass('alt') || !$button.hasClass('show-otp-popup')) {
                            $button.addClass('button alt show-otp-popup');
                        }
                    }
                }

                $(document).ready(updateButtonClass);
                $(document).ajaxComplete(updateButtonClass);
                $(document.body).on('updated_checkout payment_method_selected update_checkout', updateButtonClass);
                setInterval(updateButtonClass, 50);
            });
        </script>
        <?php
        }
    }
    add_action('wp_footer', 'add_otp_enable_forcefully');
}

// Track orders and block device if necessary
if ( ! function_exists( 'track_order_and_block_device' ) ) {
    function track_order_and_block_device($order_id) {
        if (! Helper::check_license(wp_parse_args(get_option('orderdetect_license')))) {
            return;
        }

        global $wpdb;
        $order = wc_get_order($order_id);
        $phone_number = $order->get_billing_phone();
        $device_id = isset($_COOKIE['device_id']) ? $_COOKIE['device_id'] : '';
        $ip_address = $_SERVER['REMOTE_ADDR'];
        $user_agent =  isset( $_SERVER['HTTP_USER_AGENT'] ) ? wc_clean( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
        $order_date = $order->get_date_created();

        /******************
         * Phone Information
         ******************/
        $od_phone_blocks = $wpdb->prefix . 'od_phone_blocks';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_phone_blocks (phone_number, count, blocked_until, order_id) 
            VALUES (%s, 1, NULL, %s) 
            ON DUPLICATE KEY UPDATE count = count + 1",
            $phone_number,
            $order_id
        ));

        $od_phone_device = $wpdb->prefix . 'od_phone_device';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_phone_device (phone_number, device_id, user_agent, order_id, order_created_gmt) 
            VALUES (%s, %s, %s, %s, %s)",
            $phone_number,
            $device_id,
            $user_agent,
            $order_id,
            $order_date->date('Y-m-d H:i:s')
        ));

        $od_phone_ip = $wpdb->prefix . 'od_phone_ip';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_phone_ip (phone_number, ip_address, order_id, order_created_gmt) 
            VALUES (%s, %s, %s, %s)",
            $phone_number,
            $ip_address,
            $order_id,
            $order_date->date('Y-m-d H:i:s')
        ));

        /******************
         * Device Information
         ******************/
        $od_device_blocks = $wpdb->prefix . 'od_device_blocks';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_device_blocks (device_id, count, blocked_until, order_id) 
            VALUES (%s, 1, NULL, %s) 
            ON DUPLICATE KEY UPDATE count = count + 1",
            $device_id,
            $order_id
        ));

        $od_device_phone = $wpdb->prefix . 'od_device_phone';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_device_phone (device_id, user_agent, phone_number, order_id, order_created_gmt) 
            VALUES (%s, %s, %s, %s, %s)",
            $device_id,
            $user_agent,
            $phone_number,
            $order_id,
            $order_date->date('Y-m-d H:i:s')
        ));

        $od_device_ip = $wpdb->prefix . 'od_device_ip';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_device_ip (device_id, user_agent, ip_address, order_id, order_created_gmt) 
            VALUES (%s, %s, %s, %s, %s)",
            $device_id,
            $user_agent,
            $ip_address,
            $order_id,
            $order_date->date('Y-m-d H:i:s')
        ));

        /******************
         * IP Information
         ******************/
        $od_ip_blocks = $wpdb->prefix . 'od_ip_blocks';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_ip_blocks (ip_address, count, blocked_until, order_id) 
            VALUES (%s, 1, NULL, %s) 
            ON DUPLICATE KEY UPDATE count = count + 1",
            $ip_address,
            $order_id
        ));

        $od_phone_ip = $wpdb->prefix . 'od_ip_phone';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_phone_ip (ip_address, phone_number, order_id, order_created_gmt) 
            VALUES (%s, %s, %s, %s)",
            $ip_address,
            $phone_number,
            $order_id,
            $order_date->date('Y-m-d H:i:s')
        ));

        $od_ip_device = $wpdb->prefix . 'od_ip_device';
        $wpdb->query($wpdb->prepare(
            "INSERT INTO $od_ip_device (ip_address, device_id, user_agent, order_id, order_created_gmt) 
            VALUES (%s, %s, %s, %s, %s)",
            $ip_address,
            $device_id,
            $user_agent,
            $order_id,
            $order_date->date('Y-m-d H:i:s')
        ));

        // Check and update phone block
        $phone_block_enabled = get_option('od_phone_block_enabled');
        $od_phone_block_duration = get_option('od_phone_block_duration');
        $od_phone_max_orders = get_option('od_phone_max_orders');
        
        if ($phone_block_enabled && $od_phone_max_orders > 0) {
            $od_phone_ip = $wpdb->prefix . 'od_phone_ip';
            $phone_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $od_phone_ip WHERE phone_number = %s AND order_created_gmt >= DATE_SUB(%s, INTERVAL $od_phone_block_duration MINUTE)",
                $phone_number,
                current_time('mysql', true)
            ));

            if ($phone_count >= $od_phone_max_orders) {
                $block_until = date('Y-m-d H:i:s', strtotime("+{$od_phone_block_duration} minutes"));
                $wpdb->update($od_phone_blocks, 
                    array(
                        'block_status' => 1,
                        'blocked_until' => $block_until
                    ),
                    array('phone_number' => $phone_number)
                );
            }
        }    

        // Check and update device block
        $device_block_enabled = get_option('od_device_block_enabled');
        $od_device_block_duration = get_option('od_device_block_duration');
        $od_device_max_orders = get_option('od_device_max_orders');
        
        if ($device_block_enabled && $od_device_max_orders > 0) {
            $od_device_ip = $wpdb->prefix . 'od_device_ip';
            $device_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $od_device_ip WHERE device_id = %s AND order_created_gmt >= DATE_SUB(%s, INTERVAL $od_device_block_duration MINUTE)",
                $device_id,
                current_time('mysql', true)
            ));

            if ($device_count >= $od_device_max_orders) {
                $block_until = date('Y-m-d H:i:s', strtotime("+{$od_device_block_duration} minutes"));
                $wpdb->update($od_device_blocks, 
                    array(
                        'block_status' => 1,
                        'blocked_until' => $block_until
                    ),
                    array('device_id' => $device_id)
                );
            }
        }

        // Check and update IP block
        $ip_block_enabled = get_option('od_ip_block_enabled');
        $od_ip_block_duration = get_option('od_ip_block_duration');
        $od_ip_max_orders = get_option('od_ip_max_orders');
        
        if ($ip_block_enabled && $od_ip_max_orders > 0) {
            $od_ip_phone = $wpdb->prefix . 'od_ip_phone';
            $ip_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $od_ip_phone WHERE ip_address = %s AND order_created_gmt >= DATE_SUB(%s, INTERVAL $od_ip_block_duration MINUTE)",
                $ip_address,
                current_time('mysql', true)
            ));

            if ($ip_count >= $od_ip_max_orders) {
                error_log('IP Count: ' . $ip_count);
                error_log('Max Order: ' . $od_ip_max_orders);
                $block_until = date('Y-m-d H:i:s', strtotime("+{$od_ip_block_duration} minutes"));
                $wpdb->update($od_ip_blocks, 
                    array(
                        'block_status' => 1,
                        'blocked_until' => $block_until
                    ),
                    array('ip_address' => $ip_address)
                );
            }
        }

    }
    add_action('woocommerce_thankyou', 'track_order_and_block_device', 10, 1);
}

// Set a unique device ID in a cookie if not already set
if ( ! function_exists( 'set_device_id_cookie' ) ) {
    function set_device_id_cookie() {
        if (!isset($_COOKIE['device_id']) && !headers_sent()) {
            $device_id = bin2hex(random_bytes(16)); // Generate a random device ID
            setcookie('device_id', $device_id, time() + (365 * 24 * 60 * 60), "/"); // Set cookie for 1 year
        }
    }
    // Hook the cookie setting function to WordPress init
    add_action('init', 'set_device_id_cookie');
}

if ( ! function_exists( 'order_limit_blocker_settings' ) ) {
    function order_limit_blocker_settings() {
        // Phone Number Settings
        register_setting('order-limit-blocker-settings', 'od_phone_block_enabled');
        register_setting('order-limit-blocker-settings', 'od_phone_max_orders');
        register_setting('order-limit-blocker-settings', 'od_phone_block_duration');
        register_setting('order-limit-blocker-settings', 'od_phone_block_message_title');
        register_setting('order-limit-blocker-settings', 'od_phone_block_message');

        // Device ID Settings
        register_setting('order-limit-blocker-settings', 'od_device_block_enabled');
        register_setting('order-limit-blocker-settings', 'od_device_max_orders');
        register_setting('order-limit-blocker-settings', 'od_device_block_duration');
        register_setting('order-limit-blocker-settings', 'od_device_block_message_title');
        register_setting('order-limit-blocker-settings', 'od_device_block_message');

        // IP Address Settings
        register_setting('order-limit-blocker-settings', 'od_ip_block_enabled');
        register_setting('order-limit-blocker-settings', 'od_ip_max_orders');
        register_setting('order-limit-blocker-settings', 'od_ip_block_duration');
        register_setting('order-limit-blocker-settings', 'od_ip_block_message_title');
        register_setting('order-limit-blocker-settings', 'od_ip_block_message');

        register_setting('order-limit-blocker-settings', 'od_phone_number');
        register_setting('order-limit-blocker-settings', 'od_whatsapp_link');
        register_setting('order-limit-blocker-settings', 'od_messenger_link');
        register_setting('order-limit-blocker-settings', 'od_blocker_active_tab');
        
    }
}
add_action('admin_init', 'order_limit_blocker_settings');

if ( ! function_exists( 'order_detect_courier_settings' ) ) {
    function order_detect_courier_settings() {
        // Steadfast Settings
        register_setting('order-detect-courier-settings', 'od_steadfast_api_key');
        register_setting('order-detect-courier-settings', 'od_steadfast_secret_key');

        // Pathao Settings
        register_setting('order-detect-courier-settings', 'od_pathao_api_username');
        register_setting('order-detect-courier-settings', 'od_pathao_api_password');

        // Redx Settings
        register_setting('order-detect-courier-settings', 'od_redx_api_key');

        // Paperfly Settings
        register_setting('order-detect-courier-settings', 'od_paperfly_api_username');
        register_setting('order-detect-courier-settings', 'od_paperfly_api_password');

        // Active Tab Setting
        register_setting('order-detect-courier-settings', 'od_courier_active_tab');
    }
    add_action('admin_init', 'order_detect_courier_settings');
}

if ( ! function_exists( 'make_phone_required_if_block_enabled' ) ) {
    function make_phone_required_if_block_enabled( $fields ) {
        $od_phone_block_enabled = get_option('od_phone_block_enabled');
        $fields['billing_phone']['required'] = true;
        return $fields;
    }
}
add_filter( 'woocommerce_billing_fields', 'make_phone_required_if_block_enabled', PHP_INT_MAX, 1 );

if ( ! function_exists( 'validate_billing_phone_number' ) ) {
    function validate_billing_phone_number() {
        if (isset($_POST['billing_phone']) && !empty($_POST['billing_phone'])) {
            // Use the existing normalize_customer_phone_number function to handle country code prefixes
            $billing_phone = normalize_customer_phone_number($_POST['billing_phone']);
            
            // Validate the 11-digit phone number
            if (strlen($billing_phone) != 11) {
                wc_add_notice(__('অনুগ্রহ করে একটি বৈধ ১১-ডিজিটের বাংলাদেশী ফোন নম্বর দিন।', 'order-detect'), 'error');
                return;
            }
            
            // Check if the phone number starts with '01' (Bangladeshi mobile number format)
            if (substr($billing_phone, 0, 2) !== '01') {
                wc_add_notice(__('অনুগ্রহ করে ০১ দিয়ে শুরু হওয়া একটি বৈধ বাংলাদেশী ফোন নম্বর দিন।', 'order-detect'), 'error');
            }
        }
    }
}
add_action('woocommerce_checkout_process', 'validate_billing_phone_number');

if ( ! function_exists( 'order_detect_get_remaining_block_time' ) ) {
    function order_detect_get_remaining_block_time($blocked_until) {
        if (!$blocked_until || strtotime($blocked_until) <= time()) {
            return '';
        }

        $remaining = strtotime($blocked_until) - time();
        $days = floor($remaining / 86400);
        $hours = floor(($remaining % 86400) / 3600);
        $minutes = floor(($remaining % 3600) / 60);
        
        $time_parts = [];
        if ($days > 0) $time_parts[] = $days . ' day' . ($days > 1 ? 's' : '');
        if ($hours > 0) $time_parts[] = $hours . ' hour' . ($hours > 1 ? 's' : '');
        if ($minutes > 0) $time_parts[] = $minutes . ' minute' . ($minutes > 1 ? 's' : '');
        
        return implode(', ', $time_parts);
    }
}

/*************************************
 * 
 * New Courier Score layout functions
 * 
 *************************************/
if (!function_exists('get_billing_phone_by_order_id')) {
    function get_billing_phone_by_order_id($order_id) {
        global $wpdb;

        $phone = '';

        $phone = $wpdb->get_var($wpdb->prepare(
            "SELECT phone FROM {$wpdb->prefix}wc_order_addresses WHERE order_id = %d",
            $order_id
        ));

        if(empty($phone)){
            $order = wc_get_order( $order_id );

            if ( ! $order ) {
                return false;
            }

            $billing_phone = $order->get_billing_phone();

            if ( strlen( $billing_phone ) === 13 ) {
                $billing_phone = substr( $billing_phone, 2 ); 
            }else{
                return $billing_phone;
            }

        }

        return $phone;
    }
}

if (!function_exists('steadfast_customer_details')) {
    function steadfast_customer_details($phone_number) {

        if (empty($phone_number)) {
            return null;
        }

        $phone_number = preg_replace('/[^0-9]/', '', $phone_number);
        
        // If we have credentials, proceed with the direct API call
        $url = 'https://portal.packzy.com/api/v1/fraud_check/' . $phone_number;

        $args = array(
            'method'  => 'GET'
        );

        $response = wp_remote_get($url, $args);
        $body = wp_remote_retrieve_body($response);

        $result = json_decode($body, true);
        $results = [];
        if ($result) {
            
            $results['total_parcels'] = $result['total_parcels'];
            $results['total_delivered'] = $result['total_delivered'];
            $results['total_cancelled'] = $result['total_cancelled'] ;
            $results['success_rate'] = calculate_success_rate($result['total_parcels'],$result['total_delivered']);
            
            return $results;
        }else{
            return ['total_parcels' => 0, 'total_delivered' => 0, 'total_cancelled' => 0, 'success_rate'=> 0];
        }

    }
}

if (!function_exists('pathao_authorization')) {
    function pathao_authorization() {
        $url = "https://merchant.pathao.com/api/v1/login";
        $username = get_option('od_pathao_api_username');
        $password = get_option('od_pathao_api_password');

        if (!$username || !$password) {
            error_log("Pathao API credentials not set.");
            return false;
        }

        $data = [
            "username" => $username,
            "password" => $password,
        ];

        $headers = ["Content-Type: application/json"];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true); // Enable SSL verification

        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            error_log("Pathao token fetch failed (cURL Error): " . curl_error($ch));
            curl_close($ch);
            return "Pathao token fetch failed (cURL Error): " . curl_error($ch);
        }

        $response_data = json_decode($response, true);
        curl_close($ch);

        if (!is_array($response_data) || !isset($response_data['access_token'])) {
            error_log("Invalid API response: " . $response);
            return "Invalid API response: " . $response;
        }

        update_option('od_pathao_api_token', $response_data['access_token']);
        return $response_data['access_token'];
    }
}

if (!function_exists('pathao_customer_details')) {
    function pathao_customer_details($phone_number) {
        $phone_number = preg_replace('/[^0-9]/', '', $phone_number);

        if (empty($phone_number)) {
            return null;
        }
        // If we have credentials, proceed with the direct API call
        $url = "https://merchant.pathao.com/api/v1/user/success";

        $auth_token = get_option('od_pathao_api_token');

        if (empty($auth_token)) {
            $auth_token = pathao_authorization();
            if (!$auth_token) {
                //return "Failed to retrieve new Pathao token.";
                return ['total_parcels' => 0, 'total_delivered' => 0, 'total_cancelled' => 0];

            }
        }

        $data = [
            "phone" => $phone_number
        ];

        $headers = [
            "Authorization: Bearer $auth_token",
            "Content-Type: application/json",
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true); // Enable SSL verification

        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            curl_close($ch);
            return ['total_parcels' => 0, 'total_delivered' => 0, 'total_cancelled' => 0];
        }

        $response_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $response_data = json_decode($response, true);
        curl_close($ch);

        $result = [];

        if (isset($response_data['data']['customer']) && is_array($response_data['data']['customer'])) {
            $total_parcels =  $response_data['data']['customer']['total_delivery'];
            $total_delivered =  $response_data['data']['customer']['successful_delivery'];
            $result['total_parcels'] = $total_parcels;
            $result['total_delivered'] = $total_delivered;
            $result['total_cancelled'] = $response_data['data']['customer']['total_delivery'] - $response_data['data']['customer']['successful_delivery'] ;

            $result['success_rate'] = calculate_success_rate($total_parcels, $total_delivered);
            
            return $result;
        } else {
            return ['total_parcels' => 0, 'total_delivered' => 0, 'total_cancelled' => 0];
        }
        
    }
}

if (!function_exists('redex_customer_details')) {
    function redex_customer_details($phone_number) {

        // Check if we have API credentials in the database
        $client_key = get_option('od_redx_api_key');
        
        // If we have credentials, proceed with the direct API call
        $base_url = 'https://redx.com.bd/api/redx_se/admin/parcel/customer-success-return-rate';
        $api_url = add_query_arg('phoneNumber', $phone_number, $base_url);

        $headers = [
            'Content-Type'     => 'application/json',
            'Authorization' => 'Bearer ' . $client_key,
        ];

        $response = wp_remote_get($api_url, ['headers' => $headers]);

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = json_decode(wp_remote_retrieve_body($response), true);

        $result = [];
        if (!empty($data = $response_body['data']) && is_array($response_body['data'])) {

            $result['total_parcels'] = $data['totalParcels'];
            $result['total_delivered'] = $data['deliveredParcels'];
            $result['total_cancelled'] = $data['totalParcels'] - $data['deliveredParcels'];
            $result['return_percentage'] = $data['returnPercentage'];
			
		    $result['success_rate'] = calculate_success_rate($data['totalParcels'],$data['deliveredParcels']);

            return $result;
        } else {
            return ['total_parcels' => 0, 'total_delivered' => 0, 'total_cancelled' => 0, 'return_percentage' => 0];
        }
    }
}

if (! function_exists('paperfly_courier_order_history')) {
    function paperfly_courier_order_history($phone_number) {

        if (empty($phone_number)) {
            return null;
        }
        
        // If we have credentials, proceed with the direct API call
        $phone_number = preg_replace('/[^0-9]/', '', $phone_number);
        $url = "https://go-app.paperfly.com.bd/merchant/api/react/smart-check/list.php?";
        $authorization = genarate_paperfly_login_token();

        if ($authorization == 'Unathorized') {
            return ['total_parcels' => 0, 'total_delivered' => 0, 'total_cancelled' => 0, 'status' => 'Unauthorized'];
        }

        $data = [
            "search_text" => $phone_number,
            "limit" => 50,
            "page" => 1
        ];

        $headers = [
            "Authorization: Bearer $authorization",
            "Content-Type: application/json",
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_REFERER, "https://go.paperfly.com.bd/");
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            return "Phone Number: $phone . paperfly API call failed (cURL Error): " . curl_error($ch);
        }

        $response_data = json_decode($response, true);
        curl_close($ch);
		
        $order_info = [];

        if ($response_data) {
            foreach ($response_data['records'] as $data) {
				
				$total = $data['delivered'] + $data['returned'];
                $order_info['total_parcels'] = $total;
                $order_info['total_delivered'] = $data['delivered'];
                $order_info['total_cancelled'] = $data['returned'];
				
				$order_info['success_rate'] =  calculate_success_rate($total,$data['delivered']);
            }
            return  $order_info;
        } else {
            return ['total_parcels' => 0, 'total_delivered' => 0, 'total_cancelled' => 0];
        }
    }
}

if (!function_exists('genarate_paperfly_login_token')) {
    function genarate_paperfly_login_token() {

        $username = get_option('od_paperfly_api_username', false);
        $password = get_option('od_paperfly_api_password', false);

        $url = "https://go-app.paperfly.com.bd/merchant/api/react/authentication/login_using_password.php";
        $data = [
            "username" => $username,
            "password" => $password,
        ];

        $headers = [
            "Content-Type: application/json",
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);

        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            error_log("Paperfly token fetch failed (cURL Error): " . curl_error($ch));
            return null;
        }

        $response_data = json_decode($response, true);
        curl_close($ch);

        if ($response_data['token']) {
            update_option('paperfly_api_token', $response_data['token']);
            return $response_data['token'];
        } else {
            return 'Unathorized';
        }
    }
}

if (!function_exists('calculate_success_ratio')) {
    function calculate_success_ratio($data) {
        if (empty($data)) {
            return '0.0%';
        }

        $totalParcels = 0;
        $totalDelivered = 0;

        foreach ($data as $stats) {
            if (!empty($stats['total_parcels']) && $stats['total_parcels'] > 0) {
                $totalParcels += $stats['total_parcels'];
            }

            if (!empty($stats['total_delivered']) && $stats['total_delivered'] > 0) {
                $totalDelivered += $stats['total_delivered'];
            }
        }

        if ($totalParcels > 0) {
            $totalSuccessRatio = calculate_success_rate($totalParcels,$totalDelivered);
            return $totalSuccessRatio."%"; 
        }

        return '0.0%';
    }
}

if (!function_exists('calculate_success_rate')) {
    function calculate_success_rate($total, $delivered) {
        return ($total > 0) ? round(($delivered / max(1, $total)) * 100, 2) : 0;
    }
}

if (!function_exists('neuro_orders_details')) {
    function neuro_orders_details($results,$order_id){

        $grand_total = 0;
        $grand_delivered = 0;
        $grand_cancelled = 0;

        foreach ($results as $key => $res): 
            $total = isset($res['total_parcels']) ? intval($res['total_parcels']) : 0;
            $delivered = isset($res['total_delivered']) ? intval($res['total_delivered']) : 0;
            $cancelled = isset($res['total_cancelled']) ? intval($res['total_cancelled']) : 0;

            $grand_total += $total;
            $grand_delivered += $delivered;
            $grand_cancelled += $cancelled;
        
        endforeach;

        $overall_success_ratio = ($grand_total > 0) ? calculate_success_rate($grand_total,$grand_delivered): '0';
        $order = wc_get_order($order_id);
        $mobile = '';
        
        if ($order) {
            $mobile =  $order->get_billing_phone(); 
        }
        
        ob_start(); ?>

        <div class="neuro-courier-main-container"> 
            <div class="neuro-left-panel">
                <div class="courier-total-container">
                    <div class="total">
                        <h3 id="total-title"><?php echo $grand_total; ?></h3>
                        <p>PARCELS</p>
                    </div>
                    <div class="total delivered">
                        <h3 id="neuro-delivered"><?php echo $grand_delivered; ?></h3>
                        <p>DELIVERED</p>
                    </div>
                    <div class="total canceled">
                        <h3 id="neuro-cancel"><?php echo $grand_cancelled; ?></h3>
                        <p>CANCELED</p>
                    </div>
                </div>
                <div class="all-courier-data" id="all-courier-data">
                    <div class="courier-data-header">
                        <div class="header-name">
                            <p>Courier</p>
                        </div>
                        <div class="header-name">
                            <p>Total</p>
                        </div>
                        <div class="header-name">
                            <p>Delivered</p>
                        </div>
                        <div class="header-name">
                            <p>Return</p>
                        </div>
                        <div class="header-name">
                            <p>Success</p>
                        </div>
                    </div>
                    <div class="courier-data-header courier-table">
                        <div class="header-name courier-name">
                        <img src="<?php echo ORDERDETECT_ASSETS . '/img/sfc.svg' ?>" alt="steadfast">
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['steadfast']['total_parcels']; ?></p>
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['steadfast']['total_delivered']; ?></p>
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['steadfast']['total_cancelled']; ?></p>
                        </div>
                        <div class="header-name">
                            <p><?php echo $results['steadfast']['success_rate']; ?>%</p>
                        </div>
                    </div>
                    <div class="courier-data-header courier-table">
                        <div class="header-name courier-name">
                        <img src="<?php echo ORDERDETECT_ASSETS . '/img/Pathao.svg' ?>" alt="pathao">
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['pathao']['total_parcels']; ?></p>
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['pathao']['total_delivered']; ?></p>
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['pathao']['total_cancelled']; ?></p>
                        </div>
                        <div class="header-name">
                            <p><?php echo $results['pathao']['success_rate']??0; ?>%</p>
                        </div>
                    </div>
                    <div class="courier-data-header courier-table">
                        <div class="header-name courier-name">
                        <img src="<?php echo ORDERDETECT_ASSETS . '/img/redx.svg' ?>" alt="redx">
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['redx']['total_parcels']; ?></p>
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['redx']['total_delivered']; ?></p>
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['redx']['total_cancelled']; ?></p>
                        </div>
                        <div class="header-name">
                            <p><?php echo $results['redx']['success_rate']??0; ?>%</p>
                        </div>
                    </div>
                    <div class="courier-data-header courier-table">
                        <div class="header-name courier-name">
                        <img src="<?php echo ORDERDETECT_ASSETS . '/img/paperfly.svg' ?>" alt="paperfly">
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['paperfly']['total_parcels']; ?></p>
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['paperfly']['total_delivered']; ?></p>
                        </div>
                        <div class="header-name total-count">
                            <p><?php echo $results['paperfly']['total_cancelled']; ?></p>
                        </div>
                        <div class="header-name">
                            <p><?php echo $results['paperfly']['success_rate']??0; ?>%</p>
                        </div>
                    </div>     
                </div>
            </div>

            <div class="right-panel">
                <div class="righ-panel-content">
                    <div class="neuro-logo">
                    <img src="<?php echo ORDERDETECT_ASSETS . '/img/neuro.png' ?>" alt="neuro">
                    </div>
                    <div class="progress-container">
                        <p class="right-panel-ptag">Succes Ratio</p>
                        <div class="progress-circle" data-progress="<?php echo esc_attr($overall_success_ratio); ?>"></div>
                    </div>
                    <p class="right-panel-ptag" id="requested-phone-number"><?php echo $mobile; ?></p>
                    <div>
                        <button class="neuro-refresh-data" data-order-id="<?php echo esc_attr($order_id); ?>" class="Refress-btn">Refresh Data </button>
                    </div>          
                </div>
            </div>

        </div> 

        <?php return ob_get_clean();
    }
}

if (!function_exists('courier_score_neuro_digital_api_call')) {
    function courier_score_neuro_digital_api_call( $phone_number ) {
        if ( isset( $phone_number ) && intval( $phone_number ) ) {

            $normalized_phone = normalize_customer_phone_number( $phone_number );   
            $url = 'https://dash.orderdetect.com/wp-json/courier_score/v1/delivery-tracker?phone=' . urlencode( $normalized_phone );

            $max_retries = 3;
            $retry_delay = 1;
            $attempt = 0;
            $response = null;

            do {
                $response = wp_remote_get(esc_url($url), array(
                    // 'timeout' => 90,
                ));

                if (is_wp_error($response)) {
                    error_log("API request failed: attempt ".$attempt."" . $response->get_error_message());
                    sleep($retry_delay);
                    $attempt++;
                    continue;
                }

                $body = wp_remote_retrieve_body($response);
                $courier_details = json_decode($body, true);
                if (empty($courier_details['pathao']) || 
                    (isset($courier_details['pathao']['total']) && $courier_details['pathao']['total'] == 0)) {
                    sleep($retry_delay);
                    $attempt++;
                } else {
                    break;
                }
            } while ($attempt < $max_retries);

            $body = wp_remote_retrieve_body($response);
            $courier_details = json_decode($body, true);
            
            // Transform the API response to the expected format
            $transformed_data = array();
            
            // Map the courier keys
            $courier_mapping = array(
                'stead_fast' => 'steadfast',
                'paper_fly' => 'paperfly',
                'pathao' => 'pathao',
                'redx' => 'redx'
            );
            
            // Map the field keys
            $field_mapping = array(
                'total' => 'total_parcels',
                'delivered' => 'total_delivered',
                'cancelled' => 'total_cancelled',
                'success_rate' => 'success_rate'
            );
            
            // Transform the data structure
            foreach ($courier_mapping as $api_key => $expected_key) {
                if (isset($courier_details[$api_key])) {
                    $transformed_data[$expected_key] = array();
                    
                    foreach ($field_mapping as $api_field => $expected_field) {
                        $transformed_data[$expected_key][$expected_field] = 
                            isset($courier_details[$api_key][$api_field]) ? 
                            $courier_details[$api_key][$api_field] : 0;
                    }
                }
            }
            
            // Ensure all couriers exist in the response
            foreach (array('pathao', 'steadfast', 'paperfly', 'redx') as $courier) {
                if (!isset($transformed_data[$courier])) {
                    $transformed_data[$courier] = array(
                        'total_parcels' => 0,
                        'total_delivered' => 0,
                        'total_cancelled' => 0,
                        'success_rate' => 0
                    );
                }
            }
            
            return $transformed_data;
        }
    }
}

/**
 * Add custom column widths to WooCommerce orders table
 */
if (!function_exists('custom_woocommerce_order_column_widths')) {
    function custom_woocommerce_order_column_widths() {
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->id, ['edit-shop_order', 'woocommerce_page_wc-orders'])) {
            return;
        }
        ?>
        <style type="text/css">
            .post-type-shop_order .wp-list-table .column-order_track,
            .woocommerce_page_wc-orders .wp-list-table .column-order_track {
                width: 2ch !important;
            }
        </style>
        <?php
    }
}
add_action('admin_head', 'custom_woocommerce_order_column_widths');

/**
 * Generate multi-order tracking modal HTML
 *
 * @param int $order_id The current order ID
 * @param string $full_name Customer's full name
 * @param string $email Customer's email address
 * @param string $phone Customer's phone number
 * @param array $orders_list List of WC_Order objects
 * @return string HTML output for the multi-order tracking modal
 */
if (!function_exists('generate_multi_order_tracking_modal')) {
    function generate_multi_order_tracking_modal($order_id, $full_name, $email, $phone, $orders_list) {
        // Initialize counts dynamically for all statuses
        $available_statuses = wc_get_order_statuses();
        $order_counts = ['all' => 0]; // Always include 'all'
        foreach ( $available_statuses as $status_key => $status_label ) {
            $order_counts[ str_replace( 'wc-', '', $status_key ) ] = 0; // Strip 'wc-' prefix
        }

        foreach ( $orders_list as $order ) {
            $status = $order->get_status();
            if ( isset( $order_counts[$status] ) ) {
                $order_counts[$status]++;
            }
            $order_counts['all']++;
        }

        $statuses = array_unique( array_map( function($order) { return $order->get_status(); }, $orders_list ) );
        $status_links = ['all' => $order_counts['all']]; // Include 'all' link
        foreach ( $order_counts as $status => $count ) {
            if ( $status !== 'all' && $count > 0 ) {
                $status_links[$status] = $count;
            }
        }

        ob_start();
        ?>
        <div class="order-detect-modal-dialog" id="order-detect-modal-dialog-<?php echo intval( $order_id ); ?>">
            <div class="order-detect-modal">
                <div class="order-detect-modal-content">
                    <section class="order-detect-modal-main" role="main">
                        <header class="order-detect-modal-header">
                            <h1><?php esc_html_e( 'Multi-Order Tracking', 'order-detect' ); ?></h1>
                            <button class="order-detect-modal-close order-detect-modal-close-link dashicons dashicons-no-alt">
                                <span class="screen-reader-text"><?php esc_html_e( 'Close modal panel', 'order-detect' ); ?></span>
                            </button>
                        </header>
                        <div class="wc-order-preview-addresses order-detect-preview-parent">
                            <div class="wc-order-preview-address">
                                <h2><?php esc_html_e( 'Customer details', 'order-detect' ); ?></h2>
                                <strong><?php esc_html_e( 'Name: ', 'order-detect' ); ?></strong> <?php esc_attr_e( $full_name ); ?> <br>
                                <strong><?php esc_html_e( 'Email: ', 'order-detect' ); ?></strong> <a href="mailto:<?php esc_attr_e( $email ); ?>"><?php esc_attr_e( $email ); ?></a> <br>
                                <strong><?php esc_html_e( 'Phone: ', 'order-detect' ); ?></strong> <a href="<?php esc_attr_e( $phone ); ?>"><?php esc_attr_e( $phone ); ?></a> <br>
                            </div>
                            <div class="order-status-area">
                                <ul class="subsubsub">
                                    <?php foreach ( $status_links as $status => $count ) : ?>
                                        <?php if ( $count > 0 ) : ?>
                                            <li class="<?php echo esc_attr( 'wc-' . $status ); ?>">
                                                <a href="#" class="<?php echo $status === 'all' ? 'current' : ''; ?>" data-status="<?php echo esc_attr( $status ); ?>">
                                                    <?php echo esc_html( ucfirst( str_replace( '-', ' ', $status ) ) ); ?>
                                                    <span class="count">(<?php echo esc_html( $count ); ?>)</span>
                                                </a>
                                                <?php if ( $status !== array_key_last( $status_links ) ) : ?>
                                                    |
                                                <?php endif; ?>
                                            </li>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </ul>
                            </div>    
                            <div class="order-detect-header">
                                <div class="od-header-item"><?php esc_html_e( 'Order', 'order-detect' ); ?></div>
                                <div class="od-header-item"><?php esc_html_e( 'Date', 'order-detect' ); ?></div>
                                <div class="od-header-item"><?php esc_html_e( 'Status', 'order-detect' ); ?></div>
                                <div class="od-header-item"><?php esc_html_e( 'Total', 'order-detect' ); ?></div>
                                <div class="od-header-item"><?php esc_html_e( 'Action', 'order-detect' ); ?></div>
                            </div>
                            <div class="wc-order-preview-table-wrapper" id="order-detect-preview-<?php echo $order_id; ?>">
                                <?php if ( count( $orders_list ) ): ?>
                                    <?php foreach ( $orders_list as $order ) : ?>
                                        <div class="order-detect-row order-status-<?php echo esc_attr( $order->get_status() ); ?>">
                                            <div class="od-customer-name" style="width: 100%;"><?php
                                                $buyer = '';
                                                if ( $order->get_billing_first_name() || $order->get_billing_last_name() ) {
                                                    $buyer = trim( sprintf( _x( '%1$s %2$s', 'full name', 'woocommerce' ), $order->get_billing_first_name(), $order->get_billing_last_name() ) );
                                                } elseif ( $order->get_billing_company() ) {
                                                    $buyer = trim( $order->get_billing_company() );
                                                } elseif ( $order->get_customer_id() ) {
                                                    $user  = get_user_by( 'id', $order->get_customer_id() );
                                                    $buyer = ucwords( $user->display_name );
                                                }
                                                echo '<strong>#' . esc_attr( $order->get_order_number() ) . ' ' . esc_html( $buyer ) . '</strong>'; 
                                            ?></div>
                                            <div class="od-purchase-date" style="width: 100%;"><?php echo date_i18n( 'M j, Y', strtotime( $order->get_date_created() ) ); ?></div>
                                            <div class="od-status" style="width: 100%;"><?php echo wc_get_order_status_name( $order->get_status() ); ?></div>
                                            <div class="od-purchase-price" style="width: 100%;"><?php echo wp_strip_all_tags( wc_price( $order->get_total() ) ); ?></div>
                                            <div class="od-action" style="width: 100%;">
                                                <?php 
                                                    $order_link = admin_url( 'post.php?post=' . $order->get_id() . '&action=edit' );
                                                    echo '<a href="' . esc_url( $order_link ) . '" class="button button-primary button-large od-view-order" target="_blank">' . __( "View", "order-detect" ) . '</a>';
                                                ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <div class="order-detect-row">
                                        <div class="od-no-orders"><?php esc_html_e( 'Other orders not found!', 'order-detect' ); ?></div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </section>
                </div>
            </div>
            <div class="order-detect-modal-backdrop"></div>
        </div>
        <script>
            jQuery(document).ready(function($) {
                // Cache the relevant elements
                const modalDialog = $('#order-detect-modal-dialog-<?php echo intval( $order_id ); ?>');
                const statusLinks = modalDialog.find('.order-status-area a[data-status]');
                const rowsContainer = modalDialog.find('.wc-order-preview-table-wrapper');
                const rows = rowsContainer.find('.order-detect-row');

                // Event handler for status link clicks
                statusLinks.on('click', function(e) {
                    e.preventDefault();

                    const selectedStatus = $(this).data('status');
                    if (rows.length) {
                        rows.each(function() {
                            const row = $(this);
                            // Extract the full order-status class (including hyphens)
                            const rowClassMatch = row.attr('class').match(/order-status-[\w-]+/);
                            const rowStatusClass = rowClassMatch ? rowClassMatch[0] : '';                                    
                        
                            if (selectedStatus === 'all' || rowStatusClass === `order-status-${selectedStatus}`) {
                                row.show();
                            } else {
                                row.hide();
                            }
                        });

                        // Update the 'current' class for the active status link
                        statusLinks.removeClass('current');
                        $(this).addClass('current');
                    }
                });
            });
        </script>
        <?php
        return ob_get_clean();
    }
}
//