<?php

namespace OrderDetect;

class Cronjob {
    public function register_cron_hook() {
        if (!wp_next_scheduled('od_scan_historical_orders_initial')) {
            wp_schedule_single_event(time() + 60, 'od_scan_historical_orders_initial');
        }
        
        // Also schedule the daily recurring event if it's not already scheduled
        if (!wp_next_scheduled('od_scan_historical_orders')) {
            wp_schedule_event(time(), 'daily', 'od_scan_historical_orders');
        }
    }

    public function deactivate_cron() {
        wp_clear_scheduled_hook('od_scan_historical_orders');
        wp_clear_scheduled_hook('od_scan_historical_orders_initial');    
    }

    public function scan_historical_orders() {

        global $wpdb;

        // Get orders in batches to avoid memory issues
        $batch_size = 100;
        $offset = 0;

        // Check if HPOS is enabled
        $is_hpos_enabled = get_option('woocommerce_custom_orders_table_enabled', 'no') === 'yes';

        do {
            try {
                if ($is_hpos_enabled) {
                    // Query for HPOS (High-Performance Order Storage)
                    $orders = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT DISTINCT o.id as ID, o.status as post_status 
                            FROM {$wpdb->prefix}wc_orders o
                            LEFT JOIN {$wpdb->prefix}wc_orders_meta om ON o.id = om.order_id
                            WHERE o.status IN ('pending', 'processing', 'on-hold', 'completed', 'cancelled', 'refunded', 'failed', 
                                           'wc-pending', 'wc-processing', 'wc-on-hold', 'wc-completed', 'wc-cancelled', 'wc-refunded', 'wc-failed')
                            ORDER BY o.id DESC
                            LIMIT %d OFFSET %d",
                            $batch_size,
                            $offset
                        )
                    );

                } else {
                    // Legacy query using wp_posts table
                    $orders = $wpdb->get_results(
                        $wpdb->prepare(
                            "SELECT DISTINCT p.ID, p.post_status 
                            FROM {$wpdb->posts} p
                            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
                            WHERE p.post_type = 'shop_order' 
                            AND p.post_status IN ('pending', 'processing', 'on-hold', 'completed', 'cancelled', 'refunded', 'failed',
                                               'wc-pending', 'wc-processing', 'wc-on-hold', 'wc-completed', 'wc-cancelled', 'wc-refunded', 'wc-failed')
                            ORDER BY p.ID DESC
                            LIMIT %d OFFSET %d",
                            $batch_size,
                            $offset
                        )
                    );
                    
                }
            } catch (Exception $e) {
                error_log('Query Exception: ' . $e->getMessage());
                continue;
            }

            if (!empty($orders)) {
                foreach ($orders as $order) {
                    $order_obj = wc_get_order($order->ID);
                    if (!$order_obj) continue;
                    
                    // Skip refunds as they don't have billing phone
                    if ($order_obj instanceof \Automattic\WooCommerce\Admin\Overrides\OrderRefund) continue;
                    
                    $phone_number = $order_obj->get_billing_phone();
                    if (!$phone_number) continue;

                    // Get existing order data array for this phone number
                    $existing_orders = get_post_meta($phone_number, $phone_number . '_multi_order', true);
                    $order_array = is_array($existing_orders) ? $existing_orders : array();

                    // Add order ID to the array if it doesn't exist
                    if (!in_array($order->ID, $order_array)) {
                        $order_array[] = $order->ID;
                        update_post_meta($phone_number, $phone_number . '_multi_order', $order_array);
                    }
                }
            }

            $offset += $batch_size;
        } while (!empty($orders));
    }

    public function init() {
        add_action('od_scan_historical_orders', [$this, 'scan_historical_orders']);
        add_action('od_scan_historical_orders_initial', [$this, 'scan_historical_orders']);    }
}
