<?php
namespace OrderDetect;

use OrderDetect\Helper;

class Ajax {

    private $settings;

    public function __construct() {
        $this->settings = wp_parse_args(get_option('orderdetect_settings', array(
            'sms_provider' => '',
            'sms_api_key' => array(
                'greenweb' => '',
                'alpha' => '',
                'dianahost' => '',
                'mimsms' => '',
                'bulksmsbd' => ''
            ),
            'mimsms_sender_id' => '',
            'mimsms_username' => '',
            'bulksmsbd_sender_id' => '',
            'enable_otp' => 0,
            'checkout_otp_message' => '',
            'dianahost_sender_id' => '',
            'enable_invoice' => 0,
            'enable_packing_slip' => 0,
            'invoice_packing_slip_logo' => '',
            'business_name' => '',
            'enable_footer_text' => 0,
            'footer_text_heading' => '',
            'footer_text_details' => '',
            'delivery_partner' => '',
            'primary_color' => '#000'
        )));

        add_action('wp_ajax_license_activate', array($this, 'license_activate'));
        add_action('wp_ajax_nopriv_license_activate', array($this, 'license_activate'));

        add_action('wp_ajax_license_deactivate', array($this, 'license_deactivate'));
        add_action('wp_ajax_nopriv_license_deactivate', array($this, 'license_deactivate'));

        add_action('wp_ajax_send_otp', array($this, 'send_otp'));
        add_action('wp_ajax_nopriv_send_otp', array($this, 'send_otp'));

        add_action('wp_ajax_verify_otp', array($this, 'verify_otp'));
        add_action('wp_ajax_nopriv_verify_otp', array($this, 'verify_otp'));

        add_action('wp_ajax_check_phone_is_verified', array($this, 'check_phone_is_verified'));
        add_action('wp_ajax_nopriv_check_phone_is_verified', array($this, 'check_phone_is_verified'));

        add_action('wp_ajax_get_customer_orders', array($this, 'get_customer_orders'));
        add_action('wp_ajax_nopriv_get_customer_orders', array($this, 'get_customer_orders'));

        add_action('wp_ajax_load_order_by_phone', array($this, 'load_order_by_phone'));
		add_action('wp_ajax_nopriv_load_order_by_phone', array($this, 'load_order_by_phone'));

        add_action('wp_ajax_save_sms_provider_settings', array($this, 'save_sms_provider_settings') );
        add_action('wp_ajax_nopriv_save_sms_provider_settings', array($this, 'save_sms_provider_settings') );

        add_action('wp_ajax_save_otp_setting', array($this, 'save_otp_setting') );
        add_action('wp_ajax_nopriv_save_otp_setting', array($this, 'save_otp_setting') );

        add_action('wp_ajax_save_order_alerts', array($this, 'save_order_alerts') );
        add_action('wp_ajax_nopriv_save_order_alerts', array($this, 'save_order_alerts') );

        add_action('wp_ajax_update_phone_number_status', array($this, 'update_phone_number_status') );
        add_action('wp_ajax_nopriv_update_phone_number_status', array($this, 'update_phone_number_status') );

        add_action('wp_ajax_check_phone_device_ip_is_blocked', array($this, 'check_phone_device_ip_is_blocked') );
        add_action('wp_ajax_nopriv_check_phone_device_ip_is_blocked', array($this, 'check_phone_device_ip_is_blocked') );

        add_action( 'wp_ajax_update_block_status', array( $this, 'handle_block_status_update' ) );
        add_action( 'wp_ajax_nopriv_update_block_status', array( $this, 'handle_block_status_update' ) );

        add_action( 'wp_ajax_handle_bulk_block_status', array( $this, 'handle_bulk_block_status' ) );
        add_action( 'wp_ajax_nopriv_handle_bulk_block_status', array( $this, 'handle_bulk_block_status' ) );

        add_action( 'wp_ajax_delete_block', array( $this, 'handle_delete_block' ) );
        add_action( 'wp_ajax_nopriv_delete_block', array( $this, 'handle_delete_block' ) );

        add_action( 'wp_ajax_bulk_block_delete', array( $this, 'handle_bulk_block_delete' ) );
        add_action( 'wp_ajax_nopriv_bulk_block_delete', array( $this, 'handle_bulk_block_delete' ) );

        add_action( 'wp_ajax_get_customer_info', array( $this, 'get_customer_info_ajax' ) );
        add_action( 'wp_ajax_refresh_order_info', array( $this, 'refresh_order_info' ) );

    }

    public function license_activate() {
        check_ajax_referer('order-detect-admin-nonce', 'security');

        $license_key = sanitize_text_field($_POST['license_key']);
        if (isset($license_key)) {
            $license_key = sanitize_text_field($_POST['license_key']);
            $api_params = array(
                'edd_action' => 'activate_license',
                'license'    => $license_key,
                'item_name'  => urlencode(ORDERDETECT_SL_ITEM_NAME),
                'url'        => home_url()
            );

            $response = wp_remote_post(esc_url(ORDERDETECT_STORE_URL), array( 'timeout' => 9999, 'sslverify' => false, 'body' => $api_params, 'headers' => array(
                'User-Agent' => 'Order Detect/1.0.0; ' . home_url('/')
            ) ) );

            if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code( $response )) {
                wp_send_json(array('message' => $response->get_error_message(), 'class' => 'order-detect-license-status-error'), 500);
            }

            $license_data = json_decode(wp_remote_retrieve_body($response));

            if ($license_data->success) {
                $settings = [];
                $settings['key'] = Helper::encrypt_data($license_key, ORDERDETECT_ENCRYPTION_KEY, ORDERDETECT_IV);
                $settings['expires'] = Helper::encrypt_data($license_data->expires, ORDERDETECT_ENCRYPTION_KEY, ORDERDETECT_IV);
                update_option('orderdetect_license', $settings);
                wp_send_json(array('message' => 'License activated successfully.', 'class' => 'order-detect-license-status-success'), 200);
            } else {
                wp_send_json(array('message' => 'License activation failed: '.$license_data->error, 'class' => 'order-detect-license-status-error'), 400);
            }
        } else {
            wp_send_json(array('message' => 'License key invalid!', 'class' => 'order-detect-license-status-error'), 400);
        }

        wp_die();
    }

    public function license_deactivate() {
        check_ajax_referer('order-detect-admin-nonce', 'security');
    
        $orderdetect_license = get_option('orderdetect_license');
        $license_key = isset($orderdetect_license['key']) ? $orderdetect_license['key'] : '';
        
        if ($license_key) {
            $api_params = array(
                'edd_action' => 'deactivate_license',
                'license'   => Helper::decrypt_data($license_key, ORDERDETECT_ENCRYPTION_KEY, ORDERDETECT_IV),
                'item_name' => urlencode(ORDERDETECT_SL_ITEM_NAME),
                'url'       => home_url()
            );
    
            $response = wp_remote_post(esc_url(ORDERDETECT_STORE_URL), array( 'timeout' => 9999, 'sslverify' => false, 'body' => $api_params, 'headers' => array(
                'User-Agent' => 'Order Detect/1.0.0; ' . home_url('/')
            ) ) );
    
            if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code( $response )) {
                wp_send_json(array('message' => $response->get_error_message(), 'class' => 'order-detect-license-status-error'), 500);
            }
    
            $license_data = json_decode(wp_remote_retrieve_body($response));
            if ($license_data && $license_data->success) {
                delete_option('orderdetect_license');
                wp_send_json(array('message' => 'License deactivated successfully.', 'class' => 'order-detect-license-status-success'), 200);
            } elseif ($license_data && isset($license_data->license) && $license_data->license == 'failed') {
                delete_option('orderdetect_license');
                wp_send_json(array('message' => 'License was already inactive. Local data cleaned.', 'class' => 'order-detect-license-status-warning'), 200);
            } else {
                $error_message = isset($license_data->error) ? $license_data->error : 'Unknown error';
                wp_send_json(array('message' => 'License deactivation failed: ' . $error_message, 'class' => 'order-detect-license-status-error'), 400);
            }
        } else {
            wp_send_json(array('message' => 'License key not found.', 'class' => 'order-detect-license-status-error'), 400);
        }
    
        wp_die();
    }        

    public function send_otp() {
        check_ajax_referer('order-detect-nonce', 'security');

        $phone_number = isset($_POST['phone_number']) ? sanitize_text_field($_POST['phone_number']) : '';
        $response = array();

        if (!Helper::is_valid_Bangladeshi_phone_number($phone_number)) {
            $response['success'] = false;
            $response['message'] = __('Invalid phone number format. Please enter a valid Bangladeshi phone number.', 'order-detect');
            wp_send_json($response, 400);
        }

        $otp = Helper::generate_otp($phone_number);

        if (!$otp) {
            $response['success'] = false;
            $response['message'] = __('Failed to generate OTP. Please try again later.', 'order-detect');
            wp_send_json($response, 500);
        }

        global $odSmsProvider;
        $enable = array_key_exists('enable_otp', $this->settings) ? $this->settings['enable_otp'] : '0';
        $endpoint = $odSmsProvider[$this->settings['sms_provider']]; 
        $api_key = $this->settings["sms_api_key"][$this->settings['sms_provider']];
        $checkout_otp_message = array_key_exists('checkout_otp_message', $this->settings) ? $this->settings['checkout_otp_message'] : '';

        if ( ! empty($enable) && $enable == 1 &&  ! empty($endpoint) &&  ! empty($api_key) &&  ! empty($checkout_otp_message) ) {
            $domain = $_SERVER['HTTP_HOST'];
            $message = str_replace(array('(', ')'), '', $checkout_otp_message);
            $message = str_replace('%otp_code%', $otp, $message);
            $message = str_replace('@domain', $domain, $message);

            $to = Helper::validate_and_format_phone_number( $phone_number );

            if("greenweb" === $this->settings['sms_provider'] ) {
                $endpoint .= '/api.php';
                $data = [
                    'to'=>"$to",
                    'message'=>"$message",
                    'token'=>"$api_key"
                ];
            } else if( "alpha" === $this->settings['sms_provider'] ) {
                $endpoint .= '/sendsms';
                $data = [
                    'to'=>"$to",
                    'msg'=>"$message",
                    'api_key'=>"$api_key"
                ];
            } else if( "dianahost" === $this->settings['sms_provider'] ) {
                $endpoint .= '/sms/send';
                $data = [
                    'recipient'=>"$to",
                    'type' => 'plain',
                    'sender_id' => $this->settings['dianahost_sender_id'],
                    'message'=>"$message",
                ];
            } else if( "mimsms" === $this->settings['sms_provider'] ) {
                $endpoint .= '/SmsSending/Send';
                $data = [
                    'UserName' => $this->settings['mimsms_username'],
                    'Apikey' => "$api_key",
                    'MobileNumber' => "$to",
                    'SenderName' => $this->settings['mimsms_sender_id'],
                    'TransactionType' => 'T',
                    'Message' => "$message",
                ];
                
            } else if( "bulksmsbd" === $this->settings['sms_provider'] ) {
                $endpoint .= '/smsapi';
                $data = [
                    'number'=>"$to",
                    'type' => 'text',
                    'api_key'=>"$api_key",
                    'senderid' => $this->settings['bulksmsbd_sender_id'],
                    'message'=>"$message",
                ];
            }
			
			if( "mimsms" === $this->settings['sms_provider'] ){
				file_get_contents($endpoint.'?'.http_build_query($data));
			}else{
				$ch = curl_init(); 
                curl_setopt($ch, CURLOPT_URL, $endpoint);
                curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
                curl_setopt($ch, CURLOPT_ENCODING, '');
                if( "dianahost" === $this->settings['sms_provider'] ) {
                    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                        'Accept: application/json',
                        "Authorization: Bearer $api_key",
                        "Content-Type: application/json"
                    ));
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
                } else {
                    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
                }
        
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                $smsresult = curl_exec($ch);
                curl_close($ch);
			}
            
            Helper::update_balance();
            Helper::send_sms_balance_notification();
        }

        $response['success'] = true;
        $response['message'] = sprintf(__('OTP has been sent to: %s', 'order-detect'), $phone_number);
        wp_send_json($response, 200);
        wp_die();
    }

    public function verify_otp() {
        check_ajax_referer('order-detect-nonce', 'security');

        $phone_number = isset($_POST['phone_number']) ? $_POST['phone_number'] : '';
        $otp = isset($_POST['otp']) ? sanitize_text_field($_POST['otp']) : '';

        $response = array();

        $otp_verified = Helper::verify_otp($phone_number, $otp);

        if ($otp_verified) {
            Helper::set_phone_number_verified($phone_number);
            $response['success'] = true;
            $response['message'] = 'OTP verification successful!';
        } else {
            $response['success'] = false;
            $response['message'] = 'OTP verification failed. Invalid OTP.';
        }

        wp_send_json($response, 200);
        wp_die();
    }

    public function check_phone_is_verified() {

        check_ajax_referer('order-detect-nonce', 'security');

        $phone_number = isset($_POST['phone_number']) ? $_POST['phone_number'] : '';
        $is_verified = Helper::is_phone_number_verified($phone_number);
        if ($is_verified) {
            $response['success'] = true;
            $response['message'] = 'This phone number has already been verified.';
        } else {
            $response['success'] = false;
            $response['message'] = 'This phone number is not verified.';
        }

        wp_send_json($response, 200);
    }

    public function get_customer_orders() {
        $order_id = intval($_POST['order_id']);
        $order = wc_get_order($order_id);
        
        if (!$order) {
            wp_send_json_error(array('message' => 'Invalid order ID'));
            return;
        }

        $customer_phone = $order->get_billing_phone();
        
        if (empty($customer_phone)) {
            wp_send_json_error(array('message' => 'No phone number found for this order'));
            return;
        }

        $args = array(
            'post_type' => 'shop_order',
            'post_status' => array_keys(wc_get_order_statuses()), // Include all order statuses
            'meta_query' => array(
                array(
                    'key' => '_billing_phone',
                    'value' => $customer_phone,
                    'compare' => '='
                ),
            ),
            'posts_per_page' => -1,
            'post__not_in' => array($order_id),
        );

        $orders = get_posts($args);
        
        if (empty($orders)) {
            wp_send_json_error(array('message' => 'No orders found for this customer phone number'));
            return;
        }

        $orders_data = array();
        
        foreach ($orders as $customer_order) {
            $order_id = $customer_order->ID;
            $order_date = $customer_order->post_date;
            $order_status = wc_get_order_status_name($customer_order->post_status);
            $order_total = get_post_meta($order_id, '_order_total', true);

            $orders_data[] = array(
                'id' => $order_id,
                'date' => $order_date,
                'status' => $order_status,
                'total' => wc_price($order_total),
                'edit_link' => admin_url('post.php?post=' . $order_id . '&action=edit')
            );
        }

        wp_send_json_success(array('orders' => $orders_data));

    }

   public function load_order_by_phone(){
        check_ajax_referer('order-detect-admin-nonce', 'security');
        
        if ( isset( $_POST['order_id'] ) ) {
            $order_id = intval( $_POST['order_id'] );
            $order = wc_get_order( $order_id );
            $first_name = $order->get_billing_first_name(); 
            $last_name  = $order->get_billing_last_name();
            $full_name = $first_name .' '. $last_name;
            $email     = $order->get_billing_email(); 
            $phone     = $order->get_billing_phone();
            $orders_list = get_orders_by_phone_number( $phone );
            
            if ( count($orders_list) > 1 ) {
                
                // Initialize counts dynamically for all statuses
                $output = generate_multi_order_tracking_modal($order_id, $full_name, $email, $phone, $orders_list);
                wp_send_json_success( array("html" => $output) );
            }
        }

        wp_send_json_error(array('html' => ''));
        exit;
    } 

    public function save_sms_provider_settings() {
        check_ajax_referer('order-detect-admin-nonce', 'security');
    
        $sms_provider = sanitize_text_field($_POST['sms_provider']);
        $sms_api_key = sanitize_text_field($_POST['sms_api_key']);
    
        $settings = wp_parse_args(get_option('orderdetect_settings', array(
            'sms_provider' => '',
            'sms_api_key' => array(
                'greenweb' => '',
                'alpha' => '',
                'dianahost' => '',
                'mimsms' => '',
                'bulksmsbd' => ''
            ),
            'mimsms_sender_id' => '',
            'mimsms_username' => '',
            'bulksmsbd_sender_id' => '',
            'enable_otp' => 0,
            'checkout_otp_message' => '',
            'dianahost_sender_id' => '',
            'enable_invoice' => 0,
            'enable_packing_slip' => 0,
            'invoice_packing_slip_logo' => '',
            'business_name' => '',
            'enable_footer_text' => 0,
            'footer_text_heading' => '',
            'footer_text_details' => '',
            'delivery_partner' => '',
            'primary_color' => '#000'
        )));
    
        $settings['sms_provider'] = $sms_provider;
        $settings['sms_api_key'][ $sms_provider ] = $sms_api_key;
        if( empty($sms_provider) ){
            $settings['enable_otp'] = 0;
            
            // Disable all order status notifications when no SMS provider is selected
            foreach ($settings as $key => $value) {
                if (strpos($key, 'wc-') === 0 && strpos($key, '_enable') !== false) {
                    $settings[$key] = 0;
                }
            }
        }

        if( isset($_POST['dianahost_sender_id']) ){
            $settings['dianahost_sender_id'] = sanitize_text_field($_POST['dianahost_sender_id']);
        }
        if( isset($_POST['mimsms_sender_id']) ){
            $settings['mimsms_sender_id'] = sanitize_text_field($_POST['mimsms_sender_id']);
        }
        if( isset($_POST['mimsms_username']) ){
            $settings['mimsms_username'] = sanitize_text_field($_POST['mimsms_username']);
        }
        if( isset($_POST['bulksmsbd_sender_id']) ){
            $settings['bulksmsbd_sender_id'] = sanitize_text_field($_POST['bulksmsbd_sender_id']);
        }

        update_option('orderdetect_settings', $settings);

        if (
            !empty($_POST['sms_provider']) &&
            !empty($_POST['sms_api_key'])
        ) {
            Helper::update_balance();
            $balance = get_option( 'orderdetect_sms_balance' , [
                'greenweb' => 0.00,
                'alpha' => 0.00,
                'dianahost' => 0.00,
                'mimsms' => 0.00,
                'bulksmsbd' => 0.00,
            ]);
        } 
        wp_send_json_success(array('message' => 'SMS Provider Saved Successfully!', 'balance' => ! empty( $sms_provider ) ? $balance[ $sms_provider ]: 0 ) );

    }

    public function save_otp_setting() {
        check_ajax_referer('order-detect-admin-nonce', 'security');
    
        $enable_otp = isset($_POST['enable_otp']) ? intval($_POST['enable_otp']) : 0;
        $checkout_otp_message = sanitize_textarea_field($_POST['checkout_otp_message']);
    
        $settings = wp_parse_args(get_option('orderdetect_settings', array(
            'sms_provider' => '',
            'sms_api_key' => array(
                'greenweb' => '',
                'alpha' => '',
                'dianahost' => '',
                'mimsms' => '',
                'bulksmsbd' => ''
            ),
            'mimsms_sender_id' => '',
            'mimsms_username' => '',
            'bulksmsbd_sender_id' => '',
            'enable_otp' => 0,
            'checkout_otp_message' => '',
            'dianahost_sender_id' => '',
            'enable_invoice' => 0,
            'enable_packing_slip' => 0,
            'invoice_packing_slip_logo' => '',
            'business_name' => '',
            'enable_footer_text' => 0,
            'footer_text_heading' => '',
            'footer_text_details' => '',
            'delivery_partner' => '',
            'primary_color' => '#000'
        )));
    
        if ($enable_otp) {
            // Verify SMS provider configuration
            $sms_provider = $settings['sms_provider'];
            $api_key = $settings['sms_api_key'][$sms_provider] ?? '';
            
            if (empty($sms_provider) || empty($api_key)) {
                wp_send_json_error(__('Please configure SMS provider settings before enabling OTP.', 'order-detect'));
                return;
            }
            
            $settings['enable_otp'] = $enable_otp;
        } else {
            unset($settings['enable_otp']);
        }
    
        $settings['checkout_otp_message'] = $checkout_otp_message;
        $sms_balance = get_option('orderdetect_sms_balance');
        update_option('orderdetect_settings', $settings);
        wp_send_json_success(array('message' => 'OTP Settings Saved Successfully!') );

    }

    public function save_order_alerts() {
        check_ajax_referer('order-detect-admin-nonce', 'security');
    
        $settings = wp_parse_args(get_option('orderdetect_settings', array(
            'sms_provider' => '',
            'sms_api_key' => array(
                'greenweb' => '',
                'alpha' => '',
                'dianahost' => '',
                'mimsms' => '',
                'bulksmsbd' => ''
            ),
            'mimsms_sender_id' => '',
            'mimsms_username' => '',
            'bulksmsbd_sender_id' => '',
            'enable_otp' => 0,
            'checkout_otp_message' => '',
            'dianahost_sender_id' => '',
            'enable_invoice' => 0,
            'enable_packing_slip' => 0,
            'invoice_packing_slip_logo' => '',
            'business_name' => '',
            'enable_footer_text' => 0,
            'footer_text_heading' => '',
            'footer_text_details' => '',
            'delivery_partner' => '',
            'primary_color' => '#000'
        )));
        
        if (isset($_POST['settings']) && is_array($_POST['settings'])) {
            $new_settings = $_POST['settings'];
            
            foreach ($new_settings as $key => $value) {
                $settings[$key] = sanitize_text_field($value);
            }
        }
    
        update_option('orderdetect_settings', $settings);
    
        wp_send_json_success(array('message' => 'Settings Saved Successfully!'));
    }

    public function update_phone_number_status() {

        check_ajax_referer('order-detect-admin-nonce', 'security');
        $id = isset($_POST['id']) ? intval(wp_unslash($_POST['id'])) : 0;
        $is_verified = isset($_POST['is_verified']) ? intval(wp_unslash($_POST['is_verified'])) : 0;
        if( $id > 0 && is_numeric( $id ) && in_array($is_verified, [0, 1], true) ) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'od_otp_log';
            $updated = $wpdb->update(
                $table_name,
                [ 'is_verified' => $is_verified ],
                [ 'id' => $id ],
                [ '%d' ],
                [ '%d' ]
            );
    
            if (false !== $updated) {
                wp_send_json_success([ 'message' => __('Status updated successfully.', 'order-detect') ]);
            } else {
                wp_send_json_error([ 'message' => __('Failed to update the status.', 'order-detect') ]);
            }
        } else {
            wp_send_json_error([ 'message' => __('Invalid ID or status value.', 'order-detect') ]);
        }
    }

    public function check_phone_device_ip_is_blocked() {
        check_ajax_referer('order-detect-nonce', 'security');

        global $wpdb;
        $od_phone_blocks = $wpdb->prefix . 'od_phone_blocks';
        $od_device_blocks = $wpdb->prefix . 'od_device_blocks';
        $od_ip_blocks = $wpdb->prefix . 'od_ip_blocks';
        $placeholders = array(
            '%billing_first_name%',
            '%billing_state%',
            '%billing_city%',
            '%billing_address_1%',
            '%order_number%',
            '%order_status%',
            '%order_date%',
            '%order_time%',
            '%block_duration%',
            '%max_orders%',
            '%order_count%',
            '%order_currency%',
            '%order_total%',
            '%payment_method%',
        );

        $phone_number = isset($_POST['phone_number'])? wp_unslash($_POST['phone_number']) : '';
        $device_id = isset($_POST['device_id'])? wp_unslash($_POST['device_id']) : '';
        $ip_address = $_SERVER['REMOTE_ADDR'];
        $server_user_agent = $_SERVER['HTTP_USER_AGENT'];
        $user_agent = $_POST['user_agent'];

        $is_blocked = false;
        $response = array();
        $response['is_blocked'] = $is_blocked;
        $order = null;

        $blocked_ips = get_option( 'blocked_ips', [] );
        $blocked_phones = get_option( 'blocked_phones', [] );

        // // যদি ব্লক করা আইপির তালিকায় থাকে
        if ( in_array( $ip_address, $blocked_ips ) ) {
            $response['is_blocked'] = true;
            $response['message_title'] = get_option('od_ip_block_message_title');
            $response['message'] = 'আপনার অর্ডার ব্লক করা হয়েছে সন্দেহজনক কার্যকলাপের কারণে।';
            wp_send_json($response);
        }

        // // যদি ব্লক করা ফোন নাম্বারের তালিকায় থাকে
        if ( in_array( $phone_number, $blocked_phones ) ) {
            $response['is_blocked'] = true;
            $response['message_title'] = get_option('od_phone_block_message_title');
            $response['message'] = 'আপনার ফোন নাম্বার ব্লক করা হয়েছে। অনুগ্রহ করে সঠিক তথ্য দিয়ে চেষ্টা করুন।';
            wp_send_json($response);
        }

        // Check device block
        if ($device_id && get_option('od_device_block_enabled') == 1 && get_option('od_device_max_orders') > 0) {
            $od_device_ip = $wpdb->prefix . 'od_device_ip';
            $device_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $od_device_ip WHERE device_id = %s AND order_created_gmt >= DATE_SUB(%s, INTERVAL %d MINUTE)",
                $device_id,
                current_time('mysql', true),
                get_option('od_device_block_duration')
            ));

            $device_block = $wpdb->get_row($wpdb->prepare(
                "SELECT block_status, blocked_until, order_id FROM $od_device_blocks WHERE device_id = %s",
                $device_id
            ));
            
            if ($device_count >= get_option('od_device_max_orders') && $device_block && $device_block->block_status == 1) {
                $is_blocked = true;
                $response['is_blocked'] = $is_blocked;
                
                $order = wc_get_order($device_block->order_id);
                $date_created = $order->get_date_created();
                $last_order_date = $date_created ? $date_created->format('d-m-Y') : '';
                $last_order_time = $date_created ? $date_created->format('h:i A') : '';
                $response['message_title'] = get_option('od_device_block_message_title');
                $message = str_replace(
                    $placeholders,
                    array(
                        $order->get_billing_first_name(),
                        $order->get_billing_state(),
                        $order->get_billing_city(),
                        $order->get_billing_address_1(),
                        $order->get_id(),
                        wc_get_order_status_name($order->get_status()),
                        $last_order_date,
                        $last_order_time,
                        order_detect_get_remaining_block_time($device_block->blocked_until),
                        get_option('od_device_max_orders'),
                        $device_count,
                        $order->get_currency(),
                        get_clean_price_with_currency($order->get_total()),
                        $order->get_payment_method_title(),
                    ),
                    get_option('od_device_block_message')
                );
                $response['message'] = $message;
            }
        }

        // Check IP block
        if ($ip_address && get_option('od_ip_block_enabled') == 1 && get_option('od_ip_max_orders') > 0) {
            $od_ip_orders = $wpdb->prefix . 'od_ip_phone';
            $ip_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $od_ip_orders WHERE ip_address = %s AND order_created_gmt >= DATE_SUB(%s, INTERVAL %d MINUTE)",
                $ip_address,
                current_time('mysql', true),
                get_option('od_ip_block_duration')
            ));

            $ip_block = $wpdb->get_row($wpdb->prepare(
                "SELECT block_status, blocked_until, order_id FROM $od_ip_blocks WHERE ip_address = %s",
                $ip_address
            ));
            
            if ($ip_count >= get_option('od_ip_max_orders') && $ip_block && $ip_block->block_status == 1) {
                $is_blocked = true;
                $response['is_blocked'] = $is_blocked;
                
                $order = wc_get_order($ip_block->order_id);
                $date_created = $order->get_date_created();
                $last_order_date = $date_created ? $date_created->format('d-m-Y') : '';
                $last_order_time = $date_created ? $date_created->format('h:i A') : '';
                $response['message_title'] = get_option('od_ip_block_message_title');
                $message = str_replace(
                    $placeholders,
                    array(
                        $order->get_billing_first_name(),
                        $order->get_billing_state(),
                        $order->get_billing_city(),
                        $order->get_billing_address_1(),
                        $order->get_id(),
                        wc_get_order_status_name($order->get_status()),
                        $last_order_date,
                        $last_order_time,
                        order_detect_get_remaining_block_time($ip_block->blocked_until),
                        get_option('od_ip_max_orders'),
                        $ip_count,
                        $order->get_currency(),
                        get_clean_price_with_currency($order->get_total()),
                        $order->get_payment_method_title(),
                    ),
                    get_option('od_ip_block_message')
                );
                $response['message'] = $message;
            }
        }

        $phone_block_enabled = get_option('od_phone_block_enabled');
        $od_phone_block_duration = get_option('od_phone_block_duration');
        $od_phone_max_orders = get_option('od_phone_max_orders');
        // Check phone block
        if ($phone_number && $phone_block_enabled == 1 && $od_phone_max_orders > 0) {
            $od_phone_ip = $wpdb->prefix . 'od_phone_ip';
            $phone_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $od_phone_ip WHERE phone_number = %s AND order_created_gmt >= DATE_SUB(%s, INTERVAL $od_phone_block_duration MINUTE)",
                $phone_number,
                current_time('mysql', true)
            ));
            $phone_block = $wpdb->get_row($wpdb->prepare(
                "SELECT block_status, blocked_until, order_id FROM $od_phone_blocks WHERE phone_number = %s",
                $phone_number
            ));
            
            if ( $phone_count >= $od_phone_max_orders && $phone_block && $phone_block->block_status == 1 ) {
                $is_blocked = true;
                $response['is_blocked'] = $is_blocked;
                
                $order = wc_get_order($phone_block->order_id);
                $date_created = $order->get_date_created();
                $last_order_date = $date_created ? $date_created->format('d-m-Y') : '';
                $last_order_time = $date_created ? $date_created->format('h:i A') : '';
                $order_status = $order->get_status();
                $order_number = $order->get_order_number();
                $response['message_title'] = get_option('od_phone_block_message_title');
                $message = str_replace(
                    $placeholders,
                    array(
                        $order->get_billing_first_name(),
                        $order->get_billing_state(),
                        $order->get_billing_city(),
                        $order->get_billing_address_1(),
                        $order->get_id(),
                        wc_get_order_status_name($order->get_status()),
                        $last_order_date,
                        $last_order_time,
                        order_detect_get_remaining_block_time($phone_block->blocked_until),
                        $od_phone_max_orders,
                        $phone_count,
                        $order->get_currency(),
                        get_clean_price_with_currency($order->get_total()),
                        $order->get_payment_method_title(),
                    ),
                    get_option('od_phone_block_message')
                );
                $response['message'] = $message;
            }
        }

        if ($is_blocked) {
            $response['phone_number'] = get_option('od_phone_number');
            $response['whatsapp'] = get_option('od_whatsapp_link');
            $response['messenger'] = get_option('od_messenger_link');
            $response['is_blocked'] = $is_blocked;
        }
        wp_send_json($response);

    }

    public function handle_block_status_update() {

        $identifier = sanitize_text_field($_POST['identifier']);
		$block_type = sanitize_text_field($_POST['block_type']);
		$block_until = sanitize_text_field($_POST['block_until']);

		global $wpdb;
		$table_name = $wpdb->prefix . 'od_' . $block_type . '_blocks';

		$result = false;
		if (empty($block_until)) {
			$result = $wpdb->update(
				$table_name,
				array(
					'blocked_until' => null,
					'block_status' => 0
				),
				array($block_type . ($block_type === 'device' ? '_id' : ($block_type === 'ip' ? '_address' : '_number')) => $identifier),
				array('%s', '%d'),
				array('%s')
			);
		} else {
			$result = $wpdb->update(
				$table_name,
				array(
					'blocked_until' => $block_until,
					'block_status' => 1
				),
				array($block_type . ($block_type === 'device' ? '_id' : ($block_type === 'ip' ? '_address' : '_number')) => $identifier),
				array('%s', '%d'),
				array('%s')
			);
		}

		if ($result !== false) {
			wp_send_json_success();
		} else {
			wp_send_json_error();
		}
	}

    public function handle_bulk_block_status() {
        $identifiers = isset($_POST['identifiers']) ? (array)$_POST['identifiers'] : [];
        $block_types = isset($_POST['block_types']) ? (array)$_POST['block_types'] : [];
        $block_untils = isset($_POST['block_untils']) ? (array)$_POST['block_untils'] : [];
    
        if (empty($identifiers) || count($identifiers) !== count($block_types) || count($identifiers) !== count($block_untils)) {
            wp_send_json_error('Invalid input data');
            return;
        }
    
        global $wpdb;
        $wpdb->query('START TRANSACTION');
    
        $success = true;
        $results = [];
    
        for ($i = 0; $i < count($identifiers); $i++) {
            $identifier = sanitize_text_field($identifiers[$i]);
            $block_type = sanitize_text_field($block_types[$i]);
            $block_until = sanitize_text_field($block_untils[$i]);
    
            $table_name = $wpdb->prefix . 'od_' . $block_type . '_blocks';
            $column_suffix = $block_type === 'device' ? '_id' : ($block_type === 'ip' ? '_address' : '_number');
    
            $result = $wpdb->update(
                $table_name,
                array(
                    'blocked_until' => empty($block_until) ? null : $block_until,
                    'block_status' => empty($block_until) ? 0 : 1
                ),
                array($block_type . $column_suffix => $identifier),
                array('%s', '%d'),
                array('%s')
            );
    
            if ($result === false) {
                $success = false;
                break;
            }
    
            $results[] = array(
                'identifier' => $identifier,
                'block_type' => $block_type,
                'status' => 'success'
            );
        }
    
        if ($success) {
            $wpdb->query('COMMIT');
            wp_send_json_success(array('results' => $results));
        } else {
            $wpdb->query('ROLLBACK');
            wp_send_json_error('Failed to update some blocks');
        }
    }

    public function handle_delete_block() {
        // check_ajax_referer('order-detect-admin-nonce', '_wpnonce');

        $identifier = sanitize_text_field($_POST['identifier']);
        $block_type = sanitize_text_field($_POST['block_type']);

        global $wpdb;
        $table_name = '';
        $column_name = '';
        $relation_tables = array();

        switch ($block_type) {
            case 'phone':
                $table_name = $wpdb->prefix . 'od_phone_blocks';
                $column_name = 'phone_number';
                $relation_tables = array(
                    $wpdb->prefix . 'od_phone_device',
                    $wpdb->prefix . 'od_phone_ip'
                );
                break;
            case 'device':
                $table_name = $wpdb->prefix . 'od_device_blocks';
                $column_name = 'device_id';
                $relation_tables = array(
                    $wpdb->prefix . 'od_device_phone',
                    $wpdb->prefix . 'od_device_ip'
                );
                break;
            case 'ip':
                $table_name = $wpdb->prefix . 'od_ip_blocks';
                $column_name = 'ip_address';
                $relation_tables = array(
                    $wpdb->prefix . 'od_ip_device',
                    $wpdb->prefix . 'od_ip_phone'
                );
                break;
            default:
                wp_send_json_error('Invalid block type');
                return;
        }

        $wpdb->query('START TRANSACTION');

        try {
            // Delete from relation tables first
            foreach ($relation_tables as $relation_table) {
                $wpdb->delete(
                    $relation_table,
                    array($column_name => $identifier),
                    array('%s')
                );
            }

            // Delete from the main block table
            $result = $wpdb->delete(
                $table_name,
                array($column_name => $identifier),
                array('%s')
            );

            if ($result !== false) {
                $wpdb->query('COMMIT');
                wp_send_json_success();
            } else {
                $wpdb->query('ROLLBACK');
                wp_send_json_error('Failed to delete item');
            }
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            wp_send_json_error('Failed to delete item: ' . $e->getMessage());
        }
    }

    public function handle_bulk_block_delete() {
        $identifiers = isset($_POST['identifiers']) ? (array)$_POST['identifiers'] : [];
        $block_types = isset($_POST['block_types']) ? (array)$_POST['block_types'] : [];

        if (empty($identifiers) || count($identifiers) !== count($block_types)) {
            wp_send_json_error('Invalid input data');
            return;
        }

        global $wpdb;
        $wpdb->query('START TRANSACTION');

        try {
            $success = true;
            $error_messages = [];

            for ($i = 0; $i < count($identifiers); $i++) {
                $identifier = sanitize_text_field($identifiers[$i]);
                $block_type = sanitize_text_field($block_types[$i]);

                $table_name = '';
                $column_name = '';
                $relation_table = '';
                $relation_column = '';

                switch ($block_type) {
                    case 'phone':
                        $table_name = $wpdb->prefix . 'od_phone_blocks';
                        $column_name = 'phone_number';
                        $relation_tables = [
                            $wpdb->prefix . 'od_phone_device' => 'phone_number',
                            $wpdb->prefix . 'od_phone_ip' => 'phone_number'
                        ];
                        break;
                    case 'device':
                        $table_name = $wpdb->prefix . 'od_device_blocks';
                        $column_name = 'device_id';
                        $relation_tables = [
                            $wpdb->prefix . 'od_device_phone' => 'device_id',
                            $wpdb->prefix . 'od_device_ip' => 'device_id'
                        ];
                        break;
                    case 'ip':
                        $table_name = $wpdb->prefix . 'od_ip_blocks';
                        $column_name = 'ip_address';
                        $relation_tables = [
                            $wpdb->prefix . 'od_ip_phone' => 'ip_address',
                            $wpdb->prefix . 'od_ip_device' => 'ip_address'
                        ];
                        break;
                    default:
                        $error_messages[] = "Invalid block type: {$block_type}";
                        $success = false;
                        break;
                }

                // Delete from all relation tables first
                foreach ($relation_tables as $relation_table => $relation_column) {
                    $result = $wpdb->delete(
                        $relation_table,
                        array($relation_column => $identifier),
                        array('%s')
                    );

                    if ($result === false) {
                        $error_messages[] = "Failed to delete relations from {$relation_table} for item: {$identifier}";
                        $success = false;
                        break;
                    }
                }

                if (!$success) {
                    continue;
                }

                // Then delete from blocks table
                $result = $wpdb->delete(
                    $table_name,
                    array($column_name => $identifier),
                    array('%s')
                );

                if ($result === false) {
                    $error_messages[] = "Failed to delete item: {$identifier}";
                    $success = false;
                }
            }

            if ($success) {
                $wpdb->query('COMMIT');
                wp_send_json_success();
            } else {
                $wpdb->query('ROLLBACK');
                wp_send_json_error(implode(', ', $error_messages));
            }
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            wp_send_json_error('Failed to delete items: ' . $e->getMessage());
        }
    }

    public function refresh_order_info() {
        check_ajax_referer('order-detect-admin-nonce', 'security');

        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        $query_type = isset($_POST['query_type']) ? $_POST['query_type'] : 'order';
        if( $query_type === "phone" ){
            $mobile_number = preg_replace('/[^0-9]/', '', $order_id);
        } else {
            $mobile_number = get_billing_phone_by_order_id(intval($order_id));
        }

        $od_steadfast_api_key = get_option('od_steadfast_api_key');
        $od_steadfast_secret_key = get_option('od_steadfast_secret_key');
        $od_pathao_api_username = get_option('od_pathao_api_username');
        $od_pathao_api_password = get_option('od_pathao_api_password');
        $od_paperfly_api_username = get_option('od_paperfly_api_username');
        $od_paperfly_api_password = get_option('od_paperfly_api_password');
        $od_redx_api_key = get_option('od_redx_api_key');

        $courier_data = courier_score_neuro_digital_api_call($mobile_number);
        error_log('Courier Data: '. print_r($courier_data, true));

        $steadfast = steadfast_customer_details($mobile_number);
        error_log('Steadfast Data: '. print_r($steadfast, true));

        if( empty($od_pathao_api_username) && empty($od_pathao_api_password) ) {
            $pathao = $courier_data['pathao'];
            error_log('Pathao Data: '. print_r($pathao, true));
        } else {
            $pathao = pathao_customer_details($mobile_number);
            error_log('OWN Call Pathao Data: '. print_r($pathao, true)); // Print the Pathao data for debugging or erro
        }

        if( empty($od_paperfly_api_username) && empty($od_paperfly_api_password) ) {
            $paperfly = $courier_data['paperfly'];
            error_log('Paperfly Data: '. print_r($paperfly, true)); // Print the Paperfly data for debugging or erro
        } else {
            $paperfly = paperfly_courier_order_history($mobile_number);
            error_log('OWN Call Paperfly Data: '. print_r($paperfly, true)); // Print the Paperfly data for debugging or erro
        }

        if( empty($od_redx_api_key) ) {
            $redx = $courier_data['redx'];
            error_log('Redx Data: '. print_r($redx, true)); // Print the Redx data for debugging or error
        } else {
            $redx = redex_customer_details($mobile_number);
            error_log('OWN Call Redx Data: '. print_r($redx, true)); // Print the Redx data for debugging or error
        }

        $results = [
            'pathao' => $pathao,
            'steadfast' => $steadfast,
            'paperfly' => $paperfly,
            'redx' => $redx,
        ];

        $totalParcels = 0;
        $old_parcels = 0;

        foreach ($results as $courier) {
            $totalParcels += $courier['total_parcels'] ?? 0;
        }

        $old_res = get_post_meta($mobile_number, $mobile_number . '_order_data');

        if (is_array($old_res) && array_key_exists(0, $old_res)) {
            foreach ($old_res[0] as $courier) {
                $old_parcels += $courier['total_parcels'] ?? 0;
            }
        }

        $success_ratio = calculate_success_ratio($results);
        $results['success_ratio'] = $success_ratio;

        if ($old_parcels < $totalParcels) {
            update_post_meta($mobile_number, $mobile_number . '_order_data', $results);
        }

        wp_send_json_success($results);
    }

    public function get_customer_info_ajax() {

        check_ajax_referer('order-detect-admin-nonce', 'security');

        $order_id = isset($_POST['order_id']) ? $_POST['order_id'] : 0;
        $query_type = isset($_POST['query_type']) ? $_POST['query_type']: 'order';
        if( 'phone' === $query_type ){
            $mobile_number = preg_replace('/[^0-9]/', '', $order_id);
            $data = get_post_meta($mobile_number, $mobile_number . '_order_data', true);
            wp_send_json_success($data);
        } else {
            $mobile_number = get_billing_phone_by_order_id(intval($order_id));
            $data = get_post_meta($mobile_number, $mobile_number . '_order_data', true);
            wp_send_json_success($data);
        }
        
    }

}