<?php
namespace OrderDetect\Admin;

if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/template.php';
    require_once ABSPATH . 'wp-admin/includes/class-wp-screen.php';
    require_once ABSPATH . 'wp-admin/includes/screen.php';
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class IdentifierRelationsList extends \WP_List_Table {
    private $identifier_type;
    private $identifier_value;

    public function __construct($identifier_type, $identifier_value) {
        parent::__construct([
            'singular' => 'identifier_relation',
            'plural' => 'identifier_relations',
            'ajax' => false
        ]);

        $this->identifier_type = $identifier_type;
        $this->identifier_value = $identifier_value;
    }

    public function get_columns() {
        $columns = [
            'cb' => '<input type="checkbox" />'
        ];

        $view = isset($_GET['view']) ? sanitize_text_field($_GET['view']) : '';

        switch ($view) {
            case 'phone':
                $columns['phone_number'] = __('Phone Number', 'order-detect');
                break;
            case 'device':
                $columns['device_id'] = __('Device ID', 'order-detect');
                $columns['user_agent'] = __('User Agent', 'order-detect');
                break;
            case 'ip':
                $columns['ip_address'] = __('IP Address', 'order-detect');
                break;
        }

        $columns['order_id'] = __('Order ID', 'order-detect');
        $columns['order_created_gmt'] = __('Order Date', 'order-detect');

        return $columns;
    }

    public function prepare_items() {
        global $wpdb;
        $per_page = 20;
        $current_page = $this->get_pagenum();

        // Determine tables based on identifier type and view
        switch ($this->identifier_type) {
            case 'phone':
                $device_table = $wpdb->prefix . 'od_phone_device';
                $ip_table = $wpdb->prefix . 'od_phone_ip';
                $where_column = 'phone_number';
                break;
            case 'device':
                $phone_table = $wpdb->prefix . 'od_device_phone';
                $ip_table = $wpdb->prefix . 'od_device_ip';
                $where_column = 'device_id';
                break;
            case 'ip':
                $phone_table = $wpdb->prefix . 'od_ip_phone';
                $device_table = $wpdb->prefix . 'od_ip_device';
                $where_column = 'ip_address';
                break;
        }

        // Get data based on view type
        $view = isset($_GET['view']) ? sanitize_text_field($_GET['view']) : '';
        $table_name = '';
        $select_columns = '';

        switch ($view) {
            case 'phone':
                $table_name = $phone_table;
                $select_columns = 'phone_number, order_id, order_created_gmt';
                break;
            case 'device':
                $table_name = $device_table;
                $select_columns = 'device_id, user_agent, order_id, order_created_gmt';
                break;
            case 'ip':
                $table_name = $ip_table;
                $select_columns = 'ip_address, order_id, order_created_gmt';
                break;
        }

        if (empty($table_name)) {
            $this->items = array();
            return;
        }

        $where = $wpdb->prepare(
            "{$where_column} = %s",
            $this->identifier_value
        );

        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE $where");

        $this->items = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE $where ORDER BY order_created_gmt DESC LIMIT %d OFFSET %d",
            $per_page,
            ($current_page - 1) * $per_page
        ));

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page' => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ]);

        $columns = $this->get_columns();
        $hidden = [];
        $sortable = [];
        $this->_column_headers = [$columns, $hidden, $sortable];
    }

    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'phone_number':
            case 'device_id':
            case 'ip_address':
            case 'user_agent':
                return esc_html($item->$column_name);
            case 'order_id':
                return sprintf(
                    '<a href="%s">#%s</a>',
                    esc_url(admin_url('post.php?post=' . $item->order_id . '&action=edit')),
                    esc_html($item->order_id)
                );
            case 'order_created_gmt':
                return !empty($item->order_created_gmt) ? 
                    wp_date(get_option('date_format') . ' ' . get_option('time_format'), strtotime($item->order_created_gmt)) : 
                    '';
            default:
                return print_r($item, true);
        }
    }

    public function column_cb($item) {
        return sprintf('<input type="checkbox" name="bulk-action[]" value="%s" />', $item->id);
    }

}

// Get identifier type and value from URL
$identifier_type = isset($_GET['identifier_type']) ? sanitize_text_field($_GET['identifier_type']) : '';
$identifier_value = isset($_GET['identifier_value']) ? sanitize_text_field($_GET['identifier_value']) : '';

// Initialize list table
$list_table = new IdentifierRelationsList($identifier_type, $identifier_value);
$list_table->prepare_items();
?>

<div class="wrap order-detect-blocked-devices-management">
    <h1 class="wp-heading-inline">
        <?php if (!empty($current_view) && !empty($identifier_type) && !empty($identifier_value)): ?>
            <a href="?page=block-list&tab=<?php echo esc_attr($identifier_type); ?>" class="page-title-action" style="margin-right: 10px;">
                <span class="dashicons dashicons-arrow-left-alt" style="margin-top: 4px;"></span> Back
            </a>
        <?php endif; ?>
        <?php
        switch ($identifier_type) {
            case 'phone':
                printf(__('Phone Number: %s', 'order-detect'), esc_html($identifier_value));
                break;
            case 'device':
                printf(__('Device ID: %s', 'order-detect'), esc_html($identifier_value));
                break;
            case 'ip':
                printf(__('IP Address: %s', 'order-detect'), esc_html($identifier_value));
                break;
        }
        ?>
    </h1>

    <form method="post">
        <?php $list_table->display(); ?>
    </form>
</div>