<?php

namespace OrderDetect\Admin;

if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/template.php';
    require_once ABSPATH . 'wp-admin/includes/class-wp-screen.php';
    require_once ABSPATH . 'wp-admin/includes/screen.php';
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class SMSLogList extends \WP_List_Table {

    public $phone_number;
    public $start_date;
    public $end_date;

    public function __construct($phone_number) {
        parent::__construct([
            'singular' => 'sms_log',
            'plural' => 'sms_logs',
            'ajax' => false,
        ]);
        $this->phone_number = $phone_number;
        $this->start_date = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']) : '';
        $this->end_date = isset($_GET['end_date']) ? sanitize_text_field($_GET['end_date']) : '';
    }

    public function get_columns() {
        $columns = [
            'cb' => '<input type="checkbox" />',
            'id' => __('ID', 'order-detect'),
            'phone_number' => __('Phone Number', 'order-detect'),
            'courier_score' => __('Courier Score', 'order-detect'),
            'code' => __('Code', 'order-detect'),
            'expires_at' => __('Expires At', 'order-detect'),
            'is_verified' => __('Verified?', 'order-detect'),
            'created_at' => __('Created At', 'order-detect'),
        ];
        return $columns;
    }

    protected function column_default($item, $column_name) {
        switch ($column_name) {
            case 'id':
                return $item->id;
            case 'phone_number':
                return $item->phone_number;
            case 'courier_score':
                // return '<button type="button" class="button fetch-courier-score-report" data-type="sms" data-phone_number="'.$item->phone_number.'" data-order-id="' . esc_attr($item->id) . '"><span class="dashicons dashicons-image-filter"></span> ' . __('Check Score', 'order-detect') . '</button>
                //  <div class="courier-score" id="courier-score-preview-'.$item->id.'" data-order-id="'.$item->id.'"></div>';

                $is_already_sync = get_post_meta($item->phone_number, $item->phone_number . '_order_data', true);
                $class = '';
                $btn_text = '';

                echo '<div class="neuro-handle-score">';
                    if (!empty($is_already_sync)) {
                        $success_ratio = $is_already_sync['success_ratio'];

                        $class = 'neuro-view-customer-info';
                        $btn_text = esc_html__('View', 'courier-score-check');
                        echo '<div class="neuro-success-ratio">' . $success_ratio . '</div>';
                    } else {
                        $class = 'neuro-refresh-data';
                        $btn_text = esc_html__('Refresh Score', 'courier-score-check');
                    }
                    echo '<button class="' . esc_attr($class) . '" data-order-id="' . esc_attr($item->phone_number) . '" data-query-type="phone">' . $btn_text . '</button>';
                echo '</div';    
            case 'code':
                return $item->code;
            case 'expires_at':
                $date = \DateTime::createFromFormat('Y-m-d H:i:s', $item->expires_at, new \DateTimeZone('Asia/Dhaka'));
                return $date ? $date->format('F j, Y, g:i A') : __('N/A', 'order-detect');
            case 'is_verified':
                $html = '<select name="is_verified[' . esc_attr($item->id) . ']" id="is_verified_' . esc_attr($item->id) . '" class="od_phone_number_status" data-id="' . esc_attr($item->id) . '">';
                $html .= '<option value="1"' . selected(1, $item->is_verified, false) . '>' . __('Yes', 'order-detect') . '</option>';
                $html .= '<option value="0"' . selected(0, $item->is_verified, false) . '>' . __('No', 'order-detect') . '</option>';
                $html .= '</select>';
                return $html;
            case 'created_at':
                $created_at = \DateTime::createFromFormat('Y-m-d H:i:s', $item->created_at, new \DateTimeZone('Asia/Dhaka'));
                return $created_at ? $created_at->format('F j, Y, g:i A') : __('N/A', 'order-detect');
            default:
                return print_r($item, true);
        }
    }

    protected function column_cb($item) {
        return sprintf('<input type="checkbox" name="sms_log[]" value="%s" />', $item->id);
    }

    public function get_sortable_columns() {
        $sortable_columns = [
            'id' => ['id', true],
            'phone_number' => ['phone_number', true],
            'code' => ['code', false],
            'expires_at' => ['expires_at', false],
            'is_verified' => ['is_verified', false],
            'created_at' => ['created_at', false]
        ];
        return $sortable_columns;
    }

    public function get_bulk_actions() {
        $actions = [
            'delete' => __('Delete', 'order-detect'),
            'mark_as_verified' => __('Mark as Verified', 'order-detect'),
            'mark_as_unverified' => __('Mark as Unverified', 'order-detect'),
        ];
        return $actions;
    }

    public function process_bulk_action() {
        $ids = isset($_REQUEST['sms_log']) ? $_REQUEST['sms_log'] : [];
        if (is_array($ids)) {
            $ids = array_map('intval', $ids);
        }

        if (!empty($ids)) {
            if ('delete' === $this->current_action()) {
                $this->delete_items($ids);
            } elseif ('mark_as_verified' === $this->current_action()) {
                $this->update_verification_status($ids, 1); // Mark as verified
            } elseif ('mark_as_unverified' === $this->current_action()) {
                $this->update_verification_status($ids, 0); // Mark as unverified
            }
        }
    }

    public function delete_items($ids) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'od_otp_log';
        $ids_format = implode(',', array_fill(0, count($ids), '%d'));

        $wpdb->query($wpdb->prepare("DELETE FROM $table_name WHERE id IN ($ids_format)", $ids));
    }

    public function update_verification_status($ids, $status) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'od_otp_log';
        $ids_format = implode(',', array_fill(0, count($ids), '%d'));

        $wpdb->query($wpdb->prepare(
            "UPDATE $table_name SET is_verified = %d WHERE id IN ($ids_format)",
            $status,
            ...$ids
        ));
    }

    public function get_items_per_page($option, $default = 20) {
        $user_per_page = (int) get_user_meta(get_current_user_id(), $option, true);
        if (empty($user_per_page) || $user_per_page < 1) {
            $user_per_page = $default;
        }
        return $user_per_page;
    }

    public function get_views() {
        $current_view = isset($_REQUEST['status']) ? $_REQUEST['status'] : 'all';
        $page_url = admin_url('admin.php?page=' . $_REQUEST['page']);

        $all_count = $this->get_total_items();
        $verified_count = $this->get_total_items('verified');
        $unverified_count = $this->get_total_items('unverified');

        $views = [
            'all' => sprintf(
                '<a href="%s" %s>%s</a>',
                esc_url(add_query_arg('status', 'all', $page_url)),
                ($current_view === 'all') ? 'class="current"' : '',
                __('All', 'order-detect') . ' (' . $all_count . ')'
            ),
            'verified' => sprintf(
                '<a href="%s" %s>%s</a>',
                esc_url(add_query_arg('status', 'verified', $page_url)),
                ($current_view === 'verified') ? 'class="current"' : '',
                __('Verified', 'order-detect') . ' (' . $verified_count . ')'
            ),
            'unverified' => sprintf(
                '<a href="%s" %s>%s</a>',
                esc_url(add_query_arg('status', 'unverified', $page_url)),
                ($current_view === 'unverified') ? 'class="current"' : '',
                __('Unverified', 'order-detect') . ' (' . $unverified_count . ')'
            ),
        ];

        return $views;
    }

    public function prepare_items() {
        $this->process_bulk_action();

        $status = isset($_REQUEST['status']) ? sanitize_text_field($_REQUEST['status']) : 'all'; 
        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';

        $per_page = $this->get_items_per_page('sms_log_per_page', 20);
        $current_page = $this->get_pagenum();
        $total_items = $this->get_total_items($status, $search);

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page' => $per_page,
            'total_pages' => ceil($total_items / $per_page),
        ]);

        $sms_logs = $this->get_sms_log_by_phone_number($per_page, $current_page, $status, $search );
        $this->items = $sms_logs;

        $columns = $this->get_columns();
        $hidden = [];
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = [$columns, $hidden, $sortable];
    }

    public function get_total_items($status = 'all', $search = '') {
        global $wpdb;
        $table_name = $wpdb->prefix . 'od_otp_log';

        $query = "SELECT COUNT(*) FROM $table_name WHERE 1=1";

        if (!empty($search)) {
            $query .= $wpdb->prepare(" AND phone_number LIKE %s", '%' . $wpdb->esc_like($search) . '%');
        }

        if ($status === 'verified') {
            $query .= " AND is_verified = 1";
        } elseif ($status === 'unverified') {
            $query .= " AND is_verified = 0";
        }

        // Add date range filter
        if (!empty($this->start_date)) {
            $query .= $wpdb->prepare(" AND created_at >= %s", $this->start_date . ' 00:00:00');
        }
        if (!empty($this->end_date)) {
            $query .= $wpdb->prepare(" AND created_at <= %s", $this->end_date . ' 23:59:59');
        }

        return (int) $wpdb->get_var($query);
    }

    public function get_sms_log_by_phone_number($per_page, $current_page, $status = 'all', $phone_number = '') {
        global $wpdb;
        $table_name = $wpdb->prefix . 'od_otp_log';

        $offset = ($current_page - 1) * $per_page;

        $query = "SELECT id, phone_number, code, expires_at, is_verified, created_at FROM $table_name WHERE 1=1";

        if (!empty($phone_number)) {
            $query .= $wpdb->prepare(" AND phone_number LIKE %s", '%' . $wpdb->esc_like($phone_number) . '%');
        }

        if ($status === 'verified') {
            $query .= " AND is_verified = 1";
        } elseif ($status === 'unverified') {
            $query .= " AND is_verified = 0";
        }

        // Add date range filter
        if (!empty($this->start_date)) {
            $query .= $wpdb->prepare(" AND created_at >= %s", $this->start_date . ' 00:00:00');
        }
        if (!empty($this->end_date)) {
            $query .= $wpdb->prepare(" AND created_at <= %s", $this->end_date . ' 23:59:59');
        }

        $query .= $wpdb->prepare(" ORDER BY id DESC LIMIT %d OFFSET %d", $per_page, $offset);

        return $wpdb->get_results($query);
    }

    public function extra_tablenav($which) {
        if ($which == 'top') {
            // Date filter inputs
            echo '<div class="alignleft actions">';
            echo '<input type="date" name="start_date" value="' . esc_attr($this->start_date) . '" placeholder="' . esc_attr__('Start Date', 'order-detect') . '">';
            echo '<input type="date" name="end_date" value="' . esc_attr($this->end_date) . '" placeholder="' . esc_attr__('End Date', 'order-detect') . '">';
            submit_button(__('Filter', 'order-detect'), '', 'filter_action', false);
            echo '</div>';

            // Export buttons appear here
            echo '<div class="alignleft actions bulkactions">';
                echo '<a href="' . admin_url('admin-post.php?action=export_sms_logs&status=verified') . '" class="button export-phone-number"><span class="dashicons dashicons-download"></span> ' . __('Export Verified', 'order-detect') . '</a>';
                echo '<a href="' . admin_url('admin-post.php?action=export_sms_logs&status=unverified') . '" class="button export-phone-number"><span class="dashicons dashicons-download"></span> ' . __('Export Unverified', 'order-detect') . '</a>';
            echo '</div>';
        }
    }
}
