<?php
namespace OrderDetect\Admin;

class BlockedIPList extends \WP_List_Table {
    public function __construct() {
        parent::__construct([
            'singular' => 'blocked_ip',
            'plural' => 'blocked_ips',
            'ajax' => false
        ]);
    }

    public function get_columns() {
        return [
            'cb' => '<input type="checkbox" />',
            'ip_address' => __('IP Address', 'order-detect'),
            'order_id' => __('Last Order Id', 'order-detect'),
            'count' => __('Total Orders', 'order-detect'),
            'block_until' => __('Block Until', 'order-detect'),
            'actions' => __('Actions', 'order-detect')
        ];
    }

    public function get_sortable_columns() {
        return [
            'ip_address' => ['ip_address', true],
            'order_id' => ['order_id', true],
            'count' => ['count', false],
            'block_until' => ['block_until', false]
        ];
    }

    public function prepare_items() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'od_ip_blocks';

        $per_page = 20;
        $current_page = $this->get_pagenum();
        $search = isset($_REQUEST['s']) ? sanitize_text_field($_REQUEST['s']) : '';

        $where = '1=1';
        if (!empty($search)) {
            $where .= $wpdb->prepare(' AND ip_address LIKE %s', '%' . $wpdb->esc_like($search) . '%');
        }

        $total_items = $wpdb->get_var("SELECT COUNT(*) FROM $table_name WHERE $where");

        $orderby = isset($_REQUEST['orderby']) ? sanitize_text_field($_REQUEST['orderby']) : 'id';
        $order = isset($_REQUEST['order']) ? sanitize_text_field($_REQUEST['order']) : 'DESC';

        $allowed_columns = array('id', 'ip_address', 'count', 'block_until');
        $orderby = in_array($orderby, $allowed_columns) ? $orderby : 'id';
        $order = in_array(strtoupper($order), array('ASC', 'DESC')) ? $order : 'DESC';

        $sql = $wpdb->prepare(
            "SELECT * FROM $table_name WHERE $where ORDER BY $orderby $order LIMIT %d OFFSET %d",
            $per_page,
            ($current_page - 1) * $per_page
        );
        $this->items = $wpdb->get_results($sql);

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page' => $per_page,
            'total_pages' => ceil($total_items / $per_page)
        ]);

        $columns = $this->get_columns();
        $hidden = [];
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = [$columns, $hidden, $sortable];
    }

    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'ip_address':
                return esc_html($item->ip_address);
            case 'order_id':
                if (isset($item->order_id)) {
                    $order_url = admin_url('post.php?post=' . absint($item->order_id) . '&action=edit');
                    return sprintf('<a href="%s">#%s</a>', esc_url($order_url), esc_html($item->order_id));
                }
                return 'N/A';            
            case 'count':
                return isset($item->count) ? esc_html($item->count) : '0';
            case 'block_until':
                if (isset($item->blocked_until) && strtotime($item->blocked_until) > time()) {
                    $remaining = strtotime($item->blocked_until) - time();
                    $days = floor($remaining / 86400);
                    $hours = floor(($remaining % 86400) / 3600);
                    $minutes = floor(($remaining % 3600) / 60);
                    
                    $time_parts = [];
                    if ($days > 0) $time_parts[] = $days . ' day' . ($days > 1 ? 's' : '');
                    if ($hours > 0) $time_parts[] = $hours . ' hour' . ($hours > 1 ? 's' : '');
                    if ($minutes > 0) $time_parts[] = $minutes . ' minute' . ($minutes > 1 ? 's' : '');
                    
                    return sprintf(
                        '<span class="blocked-status">Blocked (%s remaining)</span>',
                        implode(', ', $time_parts)
                    );
                }
                
                $durations = [
                    '1 hour' => '1 hour',
                    '5 hour' => '5 hours',
                    '10 hour' => '10 hours',
                    '15 hour' => '15 hours',
                    '20 hour' => '20 hours',
                    '1 day' => '1 day',
                    '2 day' => '2 days',
                    '3 day' => '3 days',
                    '5 day' => '5 days',
                    '1 week' => '1 week',
                    '2 week' => '2 weeks',
                    '1 month' => '1 month',
                    '2 month' => '2 months',
                    '3 month' => '3 months',
                    '6 month' => '6 months',
                    '1 year' => '1 year',
                    '5 year' => '5 years'
                ];
                $select = '<select class="block-duration-picker">';
                $select .= '<option value="">Select Duration</option>';
                foreach ($durations as $value => $label) {
                    $select .= sprintf(
                        '<option value="%s">%s</option>',
                        esc_attr($value),
                        esc_html($label)
                    );
                }
                $select .= '</select>';
                return $select;
            case 'actions':
                $nonce = wp_create_nonce('block_ip_' . $item->id);
                $is_blocked = isset($item->blocked_until) && strtotime($item->blocked_until) > time();
                $button_text = $is_blocked ? __('Unblock', 'order-detect') : __('Block', 'order-detect');
                $actions = sprintf(
                    '<button class="button %s block-action" data-identifier="%s" data-type="ip" data-nonce="%s">%s</button>',
                    $is_blocked ? 'button-primary' : 'button-secondary',
                    esc_attr($item->ip_address),
                    esc_attr($nonce),
                    $button_text
                );
                
                // Add Phone Number and Device buttons
                $actions .= sprintf(
                    ' <a href="?page=block-list&identifier_type=ip&identifier_value=%s&view=phone" class="button button-secondary" title="%s">Phones</a>',
                    esc_attr($item->ip_address),
                    __('View Phones', 'order-detect')
                );
                
                $actions .= sprintf(
                    ' <a href="?page=block-list&identifier_type=ip&identifier_value=%s&view=device" class="button button-secondary" title="%s">Devices</a>',
                    esc_attr($item->ip_address),
                    __('View Devices', 'order-detect')
                );

                $actions .= sprintf(
                    ' <button class="button button-secondary delete-action" data-identifier="%s" data-type="ip" data-nonce="%s" title="%s">Delete</button>',
                    esc_attr($item->ip_address),
                    esc_attr($nonce),
                    __('Delete', 'order-detect')
                );
                
                return $actions;
            default:
                return print_r($item, true);
        }
    }

    public function column_cb($item) {
        return sprintf('<input type="checkbox" name="bulk-action[]" value="%s" />', $item->id);
    }

    public function get_bulk_actions() {
        // Hide bulk actions if we're in a sub-page view
        if (isset($_GET['identifier_type']) && isset($_GET['identifier_value']) && isset($_GET['view'])) {
            return [];
        }
        return [
            'block' => __('Block', 'order-detect'),
            'unblock' => __('Unblock', 'order-detect'),
            'delete' => __('Delete', 'order-detect')
        ];
    }
}